<?php
/**
 * Plugin Name: 1 CDN Manager
 * Plugin URI: https://wp-cdn-manager.1cdn.us/
 * Description: Professional WordPress CDN integration system. Manage Image CDN, Media CDN, Static CDN, Fonts CDN, and Frontend URL with advanced caching and URL rewriting.
 * Version: 2.3.7
 * Author: Techtonic Systems Media and Research LLC
 * Author URI: https://techtonic.systems
 * License: GPLv2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: 1cdn-manager
 * Domain Path: /languages
 * Requires at least: 5.0
 * Requires PHP: 7.4
 * Tested up to: 6.8
 */

// Exit if accessed directly
if (!defined('ABSPATH')) {
    exit;
}

// Define plugin constants
define('CDN1_VERSION', '2.3.7');
define('CDN1_PLUGIN_DIR', plugin_dir_path(__FILE__));
define('CDN1_PLUGIN_URL', plugin_dir_url(__FILE__));
define('CDN1_PLUGIN_BASENAME', plugin_basename(__FILE__));

// Define WP_HOME and WP_SITEURL early if Site URL Management is enabled
// This must run BEFORE WordPress initialization
if (!defined('WP_HOME') && !defined('WP_SITEURL')) {
    $enable_site_url_management = get_option('cdn1_enable_site_url_management');
    if ($enable_site_url_management) {
        $custom_site_url = get_option('cdn1_custom_site_url');
        if (!empty($custom_site_url)) {
            $custom_site_url = rtrim($custom_site_url, '/');
            
            // Set global FIRST
            $GLOBALS['cdn1_custom_site_url'] = $custom_site_url;
            
            // Override $_SERVER variables to trick WordPress
            $parsed = parse_url($custom_site_url);
            if ($parsed) {
                $_SERVER['HTTP_HOST'] = $parsed['host'];
                $_SERVER['SERVER_NAME'] = $parsed['host'];
                if (isset($parsed['scheme'])) {
                    $_SERVER['HTTPS'] = ($parsed['scheme'] === 'https') ? 'on' : 'off';
                }
            }
            
            if (!defined('WP_HOME')) {
                define('WP_HOME', $custom_site_url);
            }
            if (!defined('WP_SITEURL')) {
                define('WP_SITEURL', $custom_site_url);
            }
            
            // Force update the options in database to match
            update_option('home', $custom_site_url);
            update_option('siteurl', $custom_site_url);
            
            // Clear WordPress object cache
            wp_cache_delete('home', 'options');
            wp_cache_delete('siteurl', 'options');
            wp_cache_delete('alloptions', 'options');
        }
    }
}

/**
 * Polyfills for PHP < 8.0
 */
if (!function_exists('str_contains')) {
    function str_contains($haystack, $needle) {
        return $needle !== '' && mb_strpos($haystack, $needle) !== false;
    }
}
if (!function_exists('str_starts_with')) {
    function str_starts_with($haystack, $needle) {
        return $needle === '' || strncmp($haystack, $needle, strlen($needle)) === 0;
    }
}

/**
 * Main Plugin Class
 */
class Utility_Server_CDN {
    
    private static $instance = null;
    
    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    private function __construct() {
        $this->load_dependencies();
        $this->init_hooks();
    }
    
    private function load_dependencies() {
        // Core classes
        require_once CDN1_PLUGIN_DIR . 'includes/class-cdn-settings.php';
        
        // Security class for domain verification and CDN key management
        require_once CDN1_PLUGIN_DIR . 'includes/class-cdn-security.php';
        
        // Site configuration MUST load very early
        require_once CDN1_PLUGIN_DIR . 'includes/class-cdn-site-config.php';
        
        // Domain mapper MUST load early to prevent redirects
        require_once CDN1_PLUGIN_DIR . 'includes/class-cdn-domain-mapper.php';
        
        // CORS handler MUST load early for proper headers
        require_once CDN1_PLUGIN_DIR . 'includes/class-cdn-cors-handler.php';
        
        require_once CDN1_PLUGIN_DIR . 'includes/class-cdn-replacer.php';
        require_once CDN1_PLUGIN_DIR . 'includes/class-cdn-cache.php';
        require_once CDN1_PLUGIN_DIR . 'includes/class-cdn-validator.php';
        require_once CDN1_PLUGIN_DIR . 'includes/class-cdn-migrator.php';
        require_once CDN1_PLUGIN_DIR . 'includes/class-cdn-content-rewriter.php';
        require_once CDN1_PLUGIN_DIR . 'includes/class-cdn-conflict-detector.php';
        require_once CDN1_PLUGIN_DIR . 'includes/class-cdn-cloudflare.php';
        
        // Admin classes (only in admin)
        if (is_admin()) {
            require_once CDN1_PLUGIN_DIR . 'includes/class-cdn-update-checker.php';
            require_once CDN1_PLUGIN_DIR . 'includes/class-cdn-admin.php';
        }
    }
    
    private function init_hooks() {
        // Activation & Deactivation
        register_activation_hook(__FILE__, [$this, 'activate']);
        register_deactivation_hook(__FILE__, [$this, 'deactivate']);
        
        // Initialize plugin
        // Initialize site config VERY early - before plugins_loaded
        add_action('muplugins_loaded', function() {
            require_once CDN1_PLUGIN_DIR . 'includes/class-cdn-settings.php';
            require_once CDN1_PLUGIN_DIR . 'includes/class-cdn-site-config.php';
            
            // Add aggressive URL filters FIRST - reuse the global set at plugin load
            $enable = get_option('cdn1_enable_site_url_management');
            
            if ($enable && !empty($GLOBALS['cdn1_custom_site_url'])) {
                $custom_url = $GLOBALS['cdn1_custom_site_url'];
                
                // Extremely aggressive: Just replace the domain part, keep the path
                add_filter('home_url', function($url, $path = '', $orig_scheme = null, $blog_id = null) {
                    $custom_url = $GLOBALS['cdn1_custom_site_url'] ?? '';
                    
                    error_log('CDN1 home_url CALLED: url=' . $url . ', custom_url=' . $custom_url);
                    
                    if (empty($custom_url)) {
                        error_log('CDN1: custom_url is EMPTY!');
                        return $url;
                    }
                    
                    if (empty($url)) {
                        error_log('CDN1: url is EMPTY!');
                        return $url;
                    }
                    
                    // Simple: just replace the domain part
                    $url_parts = parse_url($url);
                    $custom_parts = parse_url($custom_url);
                    
                    if (!$url_parts) {
                        error_log('CDN1: Failed to parse URL: ' . $url);
                        return $url;
                    }
                    
                    if (!$custom_parts) {
                        error_log('CDN1: Failed to parse custom URL: ' . $custom_url);
                        return $url;
                    }
                    
                    $old_domain = $url_parts['scheme'] . '://' . $url_parts['host'];
                    $new_domain = $custom_parts['scheme'] . '://' . $custom_parts['host'];
                    
                    $result = str_replace($old_domain, $new_domain, $url);
                    
                    error_log('CDN1 home_url REPLACE: ' . $old_domain . ' -> ' . $new_domain . ' | Result: ' . $result);
                    
                    return $result;
                }, 1, 4);
                
                add_filter('site_url', function($url, $path = '', $orig_scheme = null, $blog_id = null) {
                    $custom_url = $GLOBALS['cdn1_custom_site_url'] ?? '';
                    if (empty($custom_url) || empty($url)) return $url;
                    
                    $url_parts = parse_url($url);
                    $custom_parts = parse_url($custom_url);
                    
                    if (!$url_parts || !$custom_parts) return $url;
                    
                    $old_domain = $url_parts['scheme'] . '://' . $url_parts['host'];
                    $new_domain = $custom_parts['scheme'] . '://' . $custom_parts['host'];
                    
                    $result = str_replace($old_domain, $new_domain, $url);
                    
                    error_log('CDN1 site_url filter: ' . $url . ' -> ' . $result);
                    
                    return $result;
                }, 1, 4);
            }
            
            CDN1_Site_Config::init();
        }, 1);
        
        add_action('plugins_loaded', [$this, 'init']);
    }
    
    public function init() {
        // Initialize core functionality
        CDN1_Settings::get_instance();
        
        // Site config already initialized in muplugins_loaded hook
        
        // Domain mapper must initialize early to prevent redirects
        CDN1_Domain_Mapper::get_instance();
        
        // CORS handler must initialize early for proper headers
        CDN1_CORS_Handler::get_instance();
        
        CDN1_Replacer::get_instance();
        CDN1_Cache::get_instance();
        CDN1_Migrator::get_instance();
        CDN1_Content_Rewriter::get_instance();
        CDN1_Conflict_Detector::get_instance();
        
        // Initialize Cloudflare handler (optional integration)
        $GLOBALS['cdn1_cloudflare'] = new CDN_Cloudflare();
        
        // Initialize admin
        if (is_admin()) {
            CDN1_Update_Checker::get_instance();
            CDN1_Admin::get_instance();
        }
        
        do_action('cdn1_loaded');
    }
    
    public function activate() {
        // Set default options
        $defaults = [
            'cdn1_version' => CDN1_VERSION,
            'cdn1_enable_image_cdn' => 0,
            'cdn1_cdn_image_base' => '',
            'cdn1_enable_media_cdn' => 0,
            'cdn1_cdn_media_base' => '',
            'cdn1_enable_static_cdn' => 0,
            'cdn1_cdn_static_base' => '',
            'cdn1_enable_fonts_cdn' => 0,
            'cdn1_cdn_fonts_base' => '',
            'cdn1_enable_frontend_url' => 0,
            'cdn1_cdn_frontend_url' => '',
            'cdn1_setup_completed' => 0,
            'cdn1_cache_enabled' => 1,
        ];
        
        foreach ($defaults as $key => $value) {
            if (get_option($key) === false) {
                add_option($key, $value);
            }
        }
        
        // Initialize CORS handler to create .htaccess files
        require_once CDN1_PLUGIN_DIR . 'includes/class-cdn-cors-handler.php';
        $cors_handler = CDN1_CORS_Handler::get_instance();
        $cors_handler->ensure_cache_htaccess();
        
        // Create cache directory
        $cache_dir = WP_CONTENT_DIR . '/cache/cdn1-css';
        if (!is_dir($cache_dir)) {
            wp_mkdir_p($cache_dir);
        }
        
        // Flush rewrite rules
        flush_rewrite_rules();
        
        do_action('cdn1_activated');
    }
    
    public function deactivate() {
        // Optionally clear cache
        if (class_exists('CDN1_Cache')) {
            CDN1_Cache::clear_all_cache();
        }
        
        flush_rewrite_rules();
        
        do_action('cdn1_deactivated');
    }
}

// Initialize the plugin
Utility_Server_CDN::get_instance();