/**
 * Utility Server CDN Manager - Admin JavaScript
 */

(function($) {
    'use strict';
    
    console.log('CDN Manager Admin JS v1.9.0 loaded - AJAX Localization Fixed');
    console.log('jQuery version:', $.fn.jquery);
    
    $(document).ready(function() {
        console.log('Document ready fired');
        
        // Default CDN Toggle Handler
        $('#enable_default_cdn').on('change', function() {
            const isEnabled = $(this).is(':checked');
            
            // Show/hide Default CDN options
            $('#default-cdn-options').toggle(isEnabled);
            
            // Disable/enable custom CDN sections
            if (isEnabled) {
                $('#custom-cdn-sections').css({
                    'opacity': '0.5',
                    'pointer-events': 'none'
                });
                
                // Uncheck all custom CDN options
                $('#custom-cdn-sections input[type="checkbox"]').prop('checked', false);
                
                // Show warning
                if (!$('#default-cdn-warning').length) {
                    $('.cdn1-settings form').prepend(
                        '<div id="default-cdn-warning" class="notice notice-warning" style="margin: 20px 0; padding: 12px;">' +
                        '<p><strong>⚠️ Default CDN Enabled:</strong> All custom CDN settings have been disabled. ' +
                        'Default CDN will handle asset delivery using preconfigured 1cdn.us subdomains.</p>' +
                        '</div>'
                    );
                }
            } else {
                $('#custom-cdn-sections').css({
                    'opacity': '1',
                    'pointer-events': 'auto'
                });
                
                // Remove warning
                $('#default-cdn-warning').remove();
            }
        });
        
        // Initialize on page load
        if ($('#enable_default_cdn').is(':checked')) {
            $('#default-cdn-options').show();
            $('#custom-cdn-sections').css({
                'opacity': '0.5',
                'pointer-events': 'none'
            });
        }
        
        // Prevent form submission except for the actual Save button
        $('form.cdn1-form').on('submit', function(e) {
            // Check if the submit was triggered by the Save Settings button
            if (!$(document.activeElement).attr('name') || $(document.activeElement).attr('name') !== 'cdn1_save_settings') {
                console.log('Form submission prevented - not from Save button');
                e.preventDefault();
                return false;
            }
            console.log('Form submission allowed - from Save button');
        });
        
        // Tab Switching (Main tabs only - NOT verification tabs)
        $(document).on('click', '.cdn1-tab:not(.cdn1-verify-tab)', function(e) {
            e.preventDefault();
            e.stopPropagation();
            console.log('Main tab clicked:', $(this).data('tab'));
            
            const tabId = $(this).data('tab');
            
            // Update tab buttons (only main tabs)
            $('.cdn1-tab:not(.cdn1-verify-tab)').removeClass('active');
            $(this).addClass('active');
            
            // Update tab content
            $('.cdn1-tab-content').removeClass('active');
            $('#' + tabId).addClass('active');
            
            return false;
        });
        
        // Test CDN Button
        $('.cdn1-test-btn').on('click', function() {
            const $btn = $(this);
            const $result = $btn.siblings('.cdn1-test-result');
            const cdnUrl = $btn.data('cdn-url');
            const cdnType = $btn.data('cdn-type');
            
            if (!cdnUrl) {
                showResult($result, 'error', 'CDN URL is not configured');
                return;
            }
            
            // Show loading state
            $btn.prop('disabled', true).text(cdn1Admin.strings.testing);
            $result.removeClass('success error').addClass('loading').text('Testing connection...').show();
            
            // Make AJAX request
            $.ajax({
                url: cdn1Admin.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'cdn1_test_cdn',
                    nonce: cdn1Admin.nonce,
                    cdn_url: cdnUrl,
                    cdn_type: cdnType
                },
                success: function(response) {
                    if (response.success) {
                        showResult($result, 'success', '✓ ' + response.data.message);
                    } else {
                        showResult($result, 'error', '✗ ' + response.data.message);
                    }
                },
                error: function() {
                    showResult($result, 'error', '✗ Connection failed. Please check your configuration.');
                },
                complete: function() {
                    $btn.prop('disabled', false).text('Test Connection');
                }
            });
        });
        
        // Clear Cache Button
        $('.cdn1-clear-cache-btn').on('click', function() {
            const $btn = $(this);
            
            if (!confirm('Are you sure you want to clear all cached CSS files? This will send the X-Cache-Flush: TRUE header.')) {
                return;
            }
            
            // Show loading state
            $btn.prop('disabled', true).text(cdn1Admin.strings.clearing);
            
            // Make AJAX request
            $.ajax({
                url: cdn1Admin.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'cdn1_clear_cache',
                    nonce: cdn1Admin.nonce
                },
                success: function(response) {
                    if (response.success) {
                        alert(cdn1Admin.strings.success + ' ' + response.data.message);
                        location.reload();
                    } else {
                        alert(cdn1Admin.strings.error + ' ' + response.data.message);
                    }
                },
                error: function() {
                    alert(cdn1Admin.strings.error + ' Failed to clear cache.');
                },
                complete: function() {
                    $btn.prop('disabled', false).text('Clear All Cache');
                }
            });
        });
        
        // Helper function to show result
        function showResult($element, type, message) {
            $element
                .removeClass('loading success error')
                .addClass(type)
                .text(message)
                .show();
        }
        
        // Form validation
        $('.cdn1-form').on('submit', function(e) {
            let hasErrors = false;
            const $form = $(this);
            
            // Check if any CDN is enabled
            const imageEnabled = $form.find('input[name="enable_image_cdn"]').is(':checked');
            const staticEnabled = $form.find('input[name="enable_static_cdn"]').is(':checked');
            const linkEnabled = $form.find('input[name="enable_link_cdn"]').is(':checked');
            
            // Validate Image CDN
            if (imageEnabled) {
                const imageUrl = $form.find('input[name="cdn_image_base"]').val();
                if (!imageUrl) {
                    alert('Please enter Image CDN Base URL or disable Image CDN.');
                    hasErrors = true;
                }
            }
            
            // Validate Static CDN
            if (staticEnabled) {
                const staticUrl = $form.find('input[name="cdn_static_base"]').val();
                if (!staticUrl) {
                    alert('Please enter Static CDN Base URL or disable Static CDN.');
                    hasErrors = true;
                }
            }
            
            // Validate Link CDN
            if (linkEnabled) {
                const linkUrl = $form.find('input[name="cdn_link_base"]').val();
                if (!linkUrl) {
                    alert('Please enter Link CDN Base URL or disable Link CDN.');
                    hasErrors = true;
                }
            }
            
            if (hasErrors) {
                e.preventDefault();
                return false;
            }
        });
        
        // ===== 1CDN Security Verification & Registration =====
        
        // Note: All verification methods are now always visible on the page
        // No dropdown switching needed - simpler and more user-friendly
        console.log('CDN Security: All verification methods visible');
        console.log('CDN Security: Verify button elements found:', $('.cdn1-verify-btn').length);
        console.log('CDN Security: Auto-inject button found:', $('#cdn1-auto-inject-meta').length);
        console.log('CDN Security: Registration button found:', $('#cdn1-complete-registration').length);
        
        // Prevent form submission for security buttons
        $('.cdn1-form').on('submit', function(e) {
            // Allow form to submit normally UNLESS a security button was clicked
            if ($(document.activeElement).is('#cdn1-auto-inject-meta, .cdn1-verify-btn, #cdn1-complete-registration')) {
                console.log('Preventing form submission - security button clicked');
                e.preventDefault();
                e.stopPropagation();
                return false;
            }
        });
        
        // Auto-inject meta tag
        $(document).on('click', '#cdn1-auto-inject-meta', function(e) {
            e.preventDefault();
            e.stopPropagation();
            e.stopImmediatePropagation();
            console.log('Auto-inject meta tag clicked');
            console.log('AJAX URL:', cdn1_ajax.ajaxurl);
            console.log('Nonce:', cdn1_ajax.nonce);
            
            const button = $(this);
            button.prop('disabled', true).text('Enabling...');
            
            $.ajax({
                url: cdn1_ajax.ajaxurl,
                type: 'POST',
                data: {
                    action: 'cdn1_auto_inject_meta',
                    nonce: cdn1_ajax.nonce
                },
                success: function(response) {
                    console.log('Auto-inject response:', response);
                    if (response.success) {
                        button.text('✅ Enabled!').css('background', '#10b981');
                        setTimeout(function() {
                            button.text('Auto-Add Meta Tag').prop('disabled', false).css('background', '');
                        }, 3000);
                    } else {
                        alert('Error: ' + (response.data ? response.data.message : 'Unknown error'));
                        button.text('Auto-Add Meta Tag').prop('disabled', false);
                    }
                },
                error: function(xhr, status, error) {
                    console.error('Auto-inject error:');
                    console.error('Status:', status);
                    console.error('Error:', error);
                    console.error('Status Code:', xhr.status);
                    console.error('Response Text:', xhr.responseText);
                    
                    let errorMsg = 'Connection error';
                    if (xhr.status === 403) {
                        errorMsg = 'Permission denied (403)';
                    } else if (xhr.status === 400) {
                        errorMsg = 'Bad request (400)';
                    } else if (xhr.status === 500) {
                        errorMsg = 'Server error (500)';
                    } else if (xhr.status === 0) {
                        errorMsg = 'Network error - check if site is accessible';
                    }
                    
                    alert(errorMsg + '\nCheck browser console for details');
                    button.text('Auto-Add Meta Tag').prop('disabled', false);
                }
            });
            
            return false;
        });
        
        // Verification buttons
        $(document).on('click', '.cdn1-verify-btn', function(e) {
            e.preventDefault(); // Prevent form submission
            e.stopPropagation();
            e.stopImmediatePropagation();
            console.log('Verify button clicked');
            console.log('AJAX URL:', cdn1_ajax.ajaxurl);
            
            const button = $(this);
            const method = button.data('method');
            const resultSpan = button.siblings('.cdn1-verify-result');
            
            button.prop('disabled', true).text('Verifying...');
            resultSpan.html('<span style="color: #666;">⏳ Checking...</span>');
            
            $.ajax({
                url: cdn1_ajax.ajaxurl,
                type: 'POST',
                data: {
                    action: 'cdn1_verify_domain',
                    method: method,
                    nonce: cdn1_ajax.nonce
                },
                success: function(response) {
                    console.log('Verify response:', response);
                    if (response.success) {
                        resultSpan.html('<span style="color: #10b981; font-weight: bold;">✅ ' + response.data.message + '</span>');
                        button.text('✅ Verified!').css('background', '#10b981');
                    } else {
                        resultSpan.html('<span style="color: #dc2626; font-weight: bold;">❌ ' + (response.data ? response.data.message : 'Verification failed') + '</span>');
                        button.text('🔍 Verify ' + method.charAt(0).toUpperCase() + method.slice(1)).prop('disabled', false);
                    }
                },
                error: function(xhr, status, error) {
                    console.error('Verify error:', error);
                    console.error('Response:', xhr.responseText);
                    resultSpan.html('<span style="color: #dc2626;">❌ Connection error</span>');
                    button.text('🔍 Verify ' + method.charAt(0).toUpperCase() + method.slice(1)).prop('disabled', false);
                }
            });
            
            return false;
        });
        
        // Complete registration
        $(document).on('click', '#cdn1-complete-registration', function(e) {
            e.preventDefault();
            e.stopPropagation();
            e.stopImmediatePropagation();
            console.log('Complete registration clicked');
            
            const button = $(this);
            const resultSpan = $('#cdn1-registration-result');
            
            button.prop('disabled', true).text('Registering...');
            resultSpan.html('<span style="color: #666;">⏳ Generating key...</span>');
            
            $.ajax({
                url: cdn1_ajax.ajaxurl,
                type: 'POST',
                data: {
                    action: 'cdn1_complete_registration',
                    nonce: cdn1_ajax.nonce
                },
                success: function(response) {
                    console.log('Registration response:', response);
                    if (response.success) {
                        resultSpan.html('<span style="color: #10b981; font-weight: bold;">✅ ' + response.data.message + '</span>');
                        setTimeout(function() {
                            location.reload();
                        }, 2000);
                    } else {
                        resultSpan.html('<span style="color: #dc2626; font-weight: bold;">❌ ' + (response.data ? response.data.message : 'Registration failed') + '</span>');
                        button.text('🔐 Complete Registration & Get CDN Key').prop('disabled', false);
                    }
                },
                error: function(xhr, status, error) {
                    console.error('Registration error:', error);
                    console.error('Response:', xhr.responseText);
                    resultSpan.html('<span style="color: #dc2626;">❌ Server error: ' + (xhr.responseText || error) + '</span>');
                    button.text('🔐 Complete Registration & Get CDN Key').prop('disabled', false);
                }
            });
            
            return false;
        });
        
        // Revoke registration
        $('#cdn1-revoke-key').on('click', function() {
            // Get Default CDN status from localized data
            const defaultCdnEnabled = cdn1_ajax.default_cdn_enabled == 1;
            
            let confirmMessage = 'Are you sure you want to revoke your CDN registration?\n\n';
            
            if (defaultCdnEnabled) {
                confirmMessage += '⚠️ WARNING: This will automatically disable Default CDN and all its asset types. Your site will stop using 1cdn.us CDN immediately.\n\n';
            }
            
            confirmMessage += 'This action will remove your unlimited access and apply rate limits.';
            
            if (!confirm(confirmMessage)) {
                return;
            }
            
            const button = $(this);
            button.prop('disabled', true).text('Revoking...');
            
            $.ajax({
                url: ajaxurl,
                type: 'POST',
                data: {
                    action: 'cdn1_revoke_registration',
                    nonce: cdn1_ajax.nonce
                },
                success: function(response) {
                    if (response.success) {
                        let successMessage = response.data.message;
                        if (defaultCdnEnabled) {
                            successMessage += '\n\nDefault CDN has been automatically disabled.';
                        }
                        alert(successMessage);
                        location.reload();
                    } else {
                        alert('Error: ' + response.data.message);
                        button.text('🗑️ Revoke Registration').prop('disabled', false);
                    }
                },
                error: function() {
                    alert('Connection error');
                    button.text('🗑️ Revoke Registration').prop('disabled', false);
                }
            });
        });
        
        // Re-register domain button (in Default CDN section)
        $(document).on('click', '#cdn-reregister-btn', function(e) {
            e.preventDefault();
            e.stopPropagation();
            console.log('Re-register domain clicked');
            
            if (!confirm('Re-register your domain with 1CDN? This will refresh your registration and extend the validity period.')) {
                return;
            }
            
            const button = $(this);
            const spinner = $('#cdn-action-spinner');
            const resultDiv = $('#cdn-ownership-result');
            
            button.prop('disabled', true).text('Re-registering...');
            spinner.css('visibility', 'visible');
            resultDiv.html('');
            
            $.ajax({
                url: cdn1_ajax.ajaxurl,
                type: 'POST',
                data: {
                    action: 'cdn1_complete_registration',
                    nonce: cdn1_ajax.nonce
                },
                success: function(response) {
                    console.log('Re-registration response:', response);
                    spinner.css('visibility', 'hidden');
                    
                    if (response.success) {
                        button.text('✅ Re-registered!').css('background', '#10b981');
                        resultDiv.html('<div style="color: #10b981; font-weight: bold;">✅ ' + response.data.message + '</div>');
                        setTimeout(function() {
                            location.reload();
                        }, 1500);
                    } else {
                        resultDiv.html('<div style="color: #dc2626; font-weight: bold;">❌ ' + (response.data ? response.data.message : 'Re-registration failed') + '</div>');
                        button.text('🔄 Re-register Domain').prop('disabled', false).css('background', '');
                    }
                },
                error: function(xhr, status, error) {
                    console.error('Re-registration error:', error);
                    spinner.css('visibility', 'hidden');
                    resultDiv.html('<div style="color: #dc2626;">❌ Connection error: ' + (xhr.responseText || error) + '</div>');
                    button.text('🔄 Re-register Domain').prop('disabled', false).css('background', '');
                }
            });
            
            return false;
        });
        
        // Check ownership button (in Default CDN section)
        $(document).on('click', '#cdn-check-ownership-btn', function(e) {
            e.preventDefault();
            e.stopPropagation();
            console.log('Check ownership clicked');
            
            const button = $(this);
            const spinner = $('#cdn-action-spinner');
            const resultDiv = $('#cdn-ownership-result');
            
            // Get stored verification method from localized data
            const verificationMethod = cdn1_ajax.verification_method || '';
            
            if (!verificationMethod) {
                resultDiv.html('<div style="color: #ffc107;">⚠️ No verification method found. Please re-register your domain.</div>');
                return;
            }
            
            button.prop('disabled', true).text('Checking...');
            spinner.css('visibility', 'visible');
            resultDiv.html('');
            
            $.ajax({
                url: cdn1_ajax.ajaxurl,
                type: 'POST',
                data: {
                    action: 'cdn1_verify_domain',
                    method: verificationMethod,
                    nonce: cdn1_ajax.nonce
                },
                success: function(response) {
                    console.log('Ownership check response:', response);
                    spinner.css('visibility', 'hidden');
                    
                    if (response.success) {
                        button.text('✅ Verified!').css('color', '#10b981');
                        resultDiv.html('<div style="color: #10b981; font-weight: bold;">✅ Ownership verified successfully! Your verification records are intact.</div>');
                        setTimeout(function() {
                            button.text('🔍 Check Ownership').prop('disabled', false).css('color', '');
                        }, 3000);
                    } else {
                        button.text('🔍 Check Ownership').prop('disabled', false);
                        resultDiv.html(
                            '<div style="background: #fef3c7; padding: 15px; border-left: 4px solid #f59e0b; border-radius: 4px; margin-top: 10px;">' +
                            '<p style="margin: 0; color: #92400e; font-weight: bold;">⚠️ Ownership Verification Failed</p>' +
                            '<p style="margin: 10px 0 0 0; color: #92400e;">' + (response.data ? response.data.message : 'Unable to verify ownership') + '</p>' +
                            '<p style="margin: 10px 0 0 0; color: #92400e; font-size: 13px;">' +
                            '<strong>Action Required:</strong> Please ensure your ' + verificationMethod + ' verification is still in place, then re-register your domain.' +
                            '</p>' +
                            '</div>'
                        );
                    }
                },
                error: function(xhr, status, error) {
                    console.error('Ownership check error:', error);
                    spinner.css('visibility', 'hidden');
                    button.text('🔍 Check Ownership').prop('disabled', false);
                    resultDiv.html('<div style="color: #dc2626;">❌ Connection error: ' + (xhr.responseText || error) + '</div>');
                }
            });
            
            return false;
        });
        
    });
    
})(jQuery);
