/**
 * 1 CDN Manager - Communication System
 * Fetches and displays messages from S3
 */

(function($) {
    'use strict';
    
    const COMMUNICATIONS_URL = 'https://awsmarketplace-public-personal.s3.us-east-1.amazonaws.com/1cdn-wordpress-plugin/communications.json';
    const DISMISSED_KEY = 'cdn1_dismissed_messages';
    const SNOOZED_KEY = 'cdn1_snoozed_messages';
    
    class CommunicationsManager {
        constructor() {
            this.messages = [];
            this.activeMessages = [];
            this.currentDomain = this.getCurrentDomain();
            this.dismissedMessages = this.getDismissedMessages();
            this.snoozedMessages = this.getSnoozedMessages();
            this.overlayVisible = false;
            
            this.init();
        }
        
        init() {
            console.log('[1 CDN Comms] Starting initialization...');
            console.log('[1 CDN Comms] Current domain:', this.currentDomain);
            
            this.fetchMessages().then(() => {
                console.log('[1 CDN Comms] Messages fetched:', this.messages.length);
                this.filterActiveMessages();
                console.log('[1 CDN Comms] Active messages after filtering:', this.activeMessages.length);
                
                if (this.activeMessages.length > 0) {
                    // Check if there are new messages and clear snooze if so
                    this.clearSnoozedIfNewMessages();
                    
                    console.log('[1 CDN Comms] Rendering UI with', this.activeMessages.length, 'message(s)');
                    this.renderUI();
                    console.log('[1 CDN Comms] ✅ UI rendered successfully!');
                } else {
                    console.log('[1 CDN Comms] ⚠️ No active messages to display');
                }
            }).catch(error => {
                console.error('[1 CDN Comms] ❌ Initialization failed:', error);
            });
        }
        
        getCurrentDomain() {
            // Get WordPress site domain
            if (typeof window.location !== 'undefined') {
                return window.location.hostname;
            }
            return null;
        }
        
        fetchMessages() {
            console.log('[1 CDN Comms] Fetching fresh messages from S3...');
            
            // Add timestamp to bypass any caching
            const cacheBuster = '?t=' + Date.now();
            const urlWithCacheBuster = COMMUNICATIONS_URL + cacheBuster;
            console.log('[1 CDN Comms] URL:', urlWithCacheBuster);
            
            return fetch(urlWithCacheBuster, {
                method: 'GET',
                cache: 'no-cache',
                headers: {
                    'Cache-Control': 'no-cache',
                    'Pragma': 'no-cache'
                }
            })
            .then(response => {
                console.log('[1 CDN Comms] Response status:', response.status);
                if (!response.ok) {
                    throw new Error('HTTP ' + response.status + ' - Failed to fetch communications');
                }
                return response.json();
            })
            .then(data => {
                console.log('[1 CDN Comms] ✅ Fetch successful!');
                console.log('[1 CDN Comms] Raw data:', data);
                this.messages = data.messages || [];
                console.log('[1 CDN Comms] Loaded', this.messages.length, 'messages (no caching)');
                return this.messages;
            })
            .catch(error => {
                console.error('[1 CDN Comms] ❌ Fetch failed:', error);
                console.error('[1 CDN Comms] Error details:', error.message);
                console.error('[1 CDN Comms] This could be a CORS issue or network problem');
                this.messages = [];
                return [];
            });
        }
        
        filterActiveMessages() {
            const now = new Date();
            console.log('[1 CDN Comms] Filtering messages...');
            console.log('[1 CDN Comms] Current date:', now.toISOString());
            console.log('[1 CDN Comms] Current domain:', this.currentDomain);
            console.log('[1 CDN Comms] Dismissed message IDs:', this.dismissedMessages);
            
            this.activeMessages = this.messages.filter((msg, index) => {
                console.log(`[1 CDN Comms] Checking message ${index + 1}:`, msg.id);
                
                // Check if dismissed
                if (this.dismissedMessages.includes(msg.id)) {
                    console.log(`  ❌ Message ${msg.id} - Dismissed`);
                    return false;
                }
                
                // Check domain match
                if (msg.domain !== null && msg.domain !== this.currentDomain) {
                    console.log(`  ❌ Message ${msg.id} - Domain mismatch (needs: ${msg.domain}, have: ${this.currentDomain})`);
                    return false;
                }
                
                // Check published date
                const publishedOn = new Date(msg.data.published_on);
                if (publishedOn > now) {
                    console.log(`  ❌ Message ${msg.id} - Not yet published (publishes on: ${publishedOn.toISOString()})`);
                    return false;
                }
                
                // Check expiry date
                const expireOn = new Date(msg.data.expire_on);
                if (expireOn < now) {
                    console.log(`  ❌ Message ${msg.id} - Expired (expired on: ${expireOn.toISOString()})`);
                    return false;
                }
                
                console.log(`  ✅ Message ${msg.id} - Active (priority: ${msg.priority})`);
                return true;
            });
            
            // Sort by priority (high, medium, low)
            const priorityOrder = { high: 0, medium: 1, low: 2 };
            this.activeMessages.sort((a, b) => {
                return (priorityOrder[a.priority] || 2) - (priorityOrder[b.priority] || 2);
            });
            
            console.log('[1 CDN Comms] Active messages after filtering and sorting:', this.activeMessages.length);
            this.activeMessages.forEach((msg, i) => {
                console.log(`  ${i + 1}. ${msg.id} (${msg.priority}): ${msg.data.title || 'No title'}`);
            });
        }
        
        renderUI() {
            this.createMessageIcon();
            this.createOverlay();
            
            // Auto-show overlay if high priority message exists and is NOT snoozed
            const hasUnsnoozedHighPriority = this.activeMessages.some(msg => 
                msg.priority === 'high' && !this.snoozedMessages.includes(msg.id)
            );
            
            if (hasUnsnoozedHighPriority) {
                console.log('[1 CDN Comms] High-priority unsnoozed message found, auto-showing...');
                setTimeout(() => this.showOverlay(), 1000);
            } else if (this.activeMessages.some(msg => this.snoozedMessages.includes(msg.id))) {
                console.log('[1 CDN Comms] Messages are snoozed, not auto-showing');
            }
        }
        
        createMessageIcon() {
            const icon = $('<div>', {
                class: 'cdn1-message-icon',
                html: '<span class="dashicons dashicons-email-alt"></span><span class="cdn1-message-badge">' + this.activeMessages.length + '</span>',
                title: this.activeMessages.length + ' new message(s)'
            });
            
            icon.on('click', () => this.showOverlay());
            
            $('body').append(icon);
            
            // Add blinking animation
            setTimeout(() => icon.addClass('cdn1-blink'), 100);
        }
        
        createOverlay() {
            const overlay = $('<div>', { class: 'cdn1-overlay' });
            const modal = $('<div>', { class: 'cdn1-modal' });
            
            // Snooze button
            const snoozeBtn = $('<button>', {
                class: 'cdn1-snooze-btn',
                html: '<span class="dashicons dashicons-clock"></span> Snooze All',
                title: 'Snooze all messages (won\'t auto-show again until new messages arrive)'
            });
            
            snoozeBtn.on('click', (e) => {
                e.stopPropagation();
                this.snoozeAllMessages();
            });
            
            // Close button
            const closeBtn = $('<button>', {
                class: 'cdn1-close-btn',
                html: '<span class="dashicons dashicons-no-alt"></span>',
                title: 'Close'
            });
            
            closeBtn.on('click', (e) => {
                e.stopPropagation();
                this.hideOverlay();
            });
            
            // Header
            const header = $('<div>', {
                class: 'cdn1-modal-header',
                html: '<h2><span class="dashicons dashicons-megaphone"></span> Important Messages</h2>'
            });
            
            const headerButtons = $('<div>', { class: 'cdn1-header-buttons' });
            headerButtons.append(snoozeBtn).append(closeBtn);
            header.append(headerButtons);
            
            // Content
            const content = $('<div>', { class: 'cdn1-modal-content' });
            
            console.log('[1 CDN Comms] Creating message cards for', this.activeMessages.length, 'messages');
            this.activeMessages.forEach((msg, index) => {
                console.log(`[1 CDN Comms] Creating card ${index + 1}:`, msg.id);
                const messageCard = this.createMessageCard(msg, index);
                content.append(messageCard);
            });
            console.log('[1 CDN Comms] All cards created and appended to content');
            
            modal.append(header).append(content);
            overlay.append(modal);
            
            // Click outside to close (minimize, don't dismiss)
            overlay.on('click', (e) => {
                if ($(e.target).is('.cdn1-overlay')) {
                    this.hideOverlay();
                }
            });
            
            $('body').append(overlay);
        }
        
        createMessageCard(msg, index) {
            const card = $('<div>', {
                class: 'cdn1-message-card cdn1-priority-' + (msg.priority || 'low'),
                'data-message-id': msg.id
            });
            
            // Title
            if (msg.data.title) {
                const title = $('<h3>', {
                    class: 'cdn1-message-title',
                    text: msg.data.title
                });
                card.append(title);
            }
            
            // Body
            const body = $('<div>', {
                class: 'cdn1-message-body',
                html: msg.data.body
            });
            card.append(body);
            
            // Footer with dismiss button
            const footer = $('<div>', { class: 'cdn1-message-footer' });
            
            const dismissBtn = $('<button>', {
                class: 'cdn1-dismiss-btn',
                text: 'Dismiss',
                'data-message-id': msg.id
            });
            
            dismissBtn.on('click', () => this.dismissMessage(msg.id));
            
            // Date info
            const dateInfo = $('<span>', {
                class: 'cdn1-message-date',
                text: 'Published: ' + this.formatDate(msg.data.published_on)
            });
            
            footer.append(dateInfo).append(dismissBtn);
            card.append(footer);
            
            return card;
        }
        
        showOverlay() {
            $('.cdn1-overlay').addClass('cdn1-show');
            $('.cdn1-message-icon').removeClass('cdn1-blink');
            this.overlayVisible = true;
        }
        
        hideOverlay() {
            $('.cdn1-overlay').removeClass('cdn1-show');
            this.overlayVisible = false;
        }
        
        dismissMessage(messageId) {
            // Add to dismissed list
            this.dismissedMessages.push(messageId);
            this.saveDismissedMessages();
            
            // Remove from active messages
            this.activeMessages = this.activeMessages.filter(msg => msg.id !== messageId);
            
            // Remove card from UI
            $('.cdn1-message-card[data-message-id="' + messageId + '"]').slideUp(300, function() {
                $(this).remove();
                
                // Check if any messages left
                if ($('.cdn1-message-card').length === 0) {
                    $('.cdn1-overlay').removeClass('cdn1-show');
                    $('.cdn1-message-icon').fadeOut(300, function() {
                        $(this).remove();
                    });
                }
            });
            
            // Update badge
            const badge = $('.cdn1-message-badge');
            const count = this.activeMessages.length;
            if (count > 0) {
                badge.text(count);
            } else {
                $('.cdn1-message-icon').fadeOut(300, function() {
                    $(this).remove();
                });
            }
        }
        
        snoozeAllMessages() {
            console.log('[1 CDN Comms] Snoozing all current messages');
            
            // Add all active message IDs to snoozed list
            this.activeMessages.forEach(msg => {
                if (!this.snoozedMessages.includes(msg.id)) {
                    this.snoozedMessages.push(msg.id);
                }
            });
            
            // Save to localStorage
            try {
                localStorage.setItem(SNOOZED_KEY, JSON.stringify(this.snoozedMessages));
                console.log('[1 CDN Comms] Snoozed message IDs:', this.snoozedMessages);
            } catch (e) {
                console.error('[1 CDN Comms] Failed to save snoozed messages');
            }
            
            // Hide overlay
            this.hideOverlay();
            
            // Show confirmation (brief notification)
            this.showSnoozeConfirmation();
        }
        
        showSnoozeConfirmation() {
            const notification = $('<div>', {
                class: 'cdn1-snooze-notification',
                html: '<span class="dashicons dashicons-yes"></span> Messages snoozed. They won\'t auto-show until new messages arrive.',
                css: {
                    position: 'fixed',
                    top: '50px',
                    right: '20px',
                    background: '#667eea',
                    color: 'white',
                    padding: '15px 20px',
                    borderRadius: '8px',
                    boxShadow: '0 4px 12px rgba(0,0,0,0.3)',
                    zIndex: 100001,
                    display: 'flex',
                    alignItems: 'center',
                    gap: '10px',
                    fontSize: '14px',
                    fontWeight: '600'
                }
            });
            
            $('body').append(notification);
            
            // Fade out after 3 seconds
            setTimeout(() => {
                notification.fadeOut(500, function() {
                    $(this).remove();
                });
            }, 3000);
        }
        
        getSnoozedMessages() {
            try {
                const snoozed = localStorage.getItem(SNOOZED_KEY);
                return snoozed ? JSON.parse(snoozed) : [];
            } catch (e) {
                return [];
            }
        }
        
        clearSnoozedIfNewMessages() {
            // If there are new message IDs not in the snoozed list, clear snooze
            const newMessages = this.activeMessages.filter(msg => !this.snoozedMessages.includes(msg.id));
            
            if (newMessages.length > 0) {
                console.log('[1 CDN Comms] New messages detected, clearing snooze');
                localStorage.removeItem(SNOOZED_KEY);
                this.snoozedMessages = [];
            }
        }
        
        formatDate(dateString) {
            const date = new Date(dateString);
            return date.toLocaleDateString('en-US', {
                year: 'numeric',
                month: 'short',
                day: 'numeric'
            });
        }
        
        getDismissedMessages() {
            try {
                const dismissed = localStorage.getItem(DISMISSED_KEY);
                return dismissed ? JSON.parse(dismissed) : [];
            } catch (e) {
                return [];
            }
        }
        
        saveDismissedMessages() {
            try {
                localStorage.setItem(DISMISSED_KEY, JSON.stringify(this.dismissedMessages));
            } catch (e) {
                console.warn('Could not save dismissed messages');
            }
        }
    }
    
    // Initialize when DOM is ready
    $(document).ready(function() {
        // Initialize on ALL WordPress admin pages
        // The system will still filter messages by domain
        if ($('body').hasClass('wp-admin') || 
            $('body').hasClass('toplevel_page_cdn1-settings') || 
            $('.wrap.cdn1-settings').length > 0 ||
            window.location.href.indexOf('wp-admin') !== -1) {
            
            console.log('[1 CDN Comms] Initializing communication system...');
            console.log('[1 CDN Comms] Current URL:', window.location.href);
            new CommunicationsManager();
        } else {
            console.log('[1 CDN Comms] Not on admin page, skipping initialization');
        }
    });
    
    // Manual refresh function (can be called from console)
    window.refreshCommunications = function() {
        localStorage.removeItem(CACHE_KEY);
        console.log('[1 CDN Comms] Cache cleared, reloading...');
        location.reload();
    };
    
    // Debug function
    window.debugCommunications = function() {
        console.log('=== Communication System Debug ===');
        console.log('Dismissed:', localStorage.getItem(DISMISSED_KEY));
        console.log('Snoozed:', localStorage.getItem(SNOOZED_KEY));
        console.log('Current domain:', window.location.hostname);
        console.log('Body classes:', $('body').attr('class'));
        console.log('Is WordPress Admin:', $('body').hasClass('wp-admin'));
    };
    
})(jQuery);
