<?php
/**
 * Site Configuration Admin Page
 *
 * @package CDN1_Manager
 * @since 1.4.0
 */

if (!defined('ABSPATH')) {
    exit;
}

// Get settings - these should always work
$feature_enabled = get_option('cdn1_enable_site_url_management', 0);
$custom_site_url = get_option('cdn1_custom_site_url', '');
$custom_admin_url = get_option('cdn1_custom_admin_url', '');

// Check if Default CDN is enabled
$default_cdn_enabled = get_option('cdn1_enable_default_cdn', 0);

// If Default CDN is enabled, force disable Site URL Management
if ($default_cdn_enabled) {
    $feature_enabled = 0;
}
?>

<div class="wrap">
    <?php include CDN1_PLUGIN_DIR . 'admin/views/branding.php'; ?>
    
    <h1>
        <span class="dashicons dashicons-admin-site" style="font-size: 32px; width: 32px; height: 32px;"></span>
        <?php esc_html_e('Site URL Configuration', '1cdn-manager'); ?>
    </h1>
    
    <p class="description">
        <?php esc_html_e('Manage your WordPress site URLs to prevent plugins from detecting wrong URLs when admin is accessed from a different domain.', '1cdn-manager'); ?>
    </p>
    
    <?php 
    // Check for configuration issues - do early check here
    $issues = [];
    if (class_exists('CDN1_Site_Config') && method_exists('CDN1_Site_Config', 'check_configuration')) {
        $check_result = CDN1_Site_Config::check_configuration();
        if (is_array($check_result)) {
            // Filter out warnings about WP_HOME and WP_SITEURL constants
            foreach ($check_result as $issue) {
                // Skip warnings about WP_HOME and WP_SITEURL being defined
                if (isset($issue['message']) && 
                    (strpos($issue['message'], 'WP_HOME is defined') === false && 
                     strpos($issue['message'], 'WP_SITEURL is defined') === false)) {
                    $issues[] = $issue;
                }
            }
        }
    }
    ?>
    
    <?php if (!empty($issues)): ?>
        <div class="cdn1-diagnostics-issues">
            <?php foreach ($issues as $issue): ?>
                <div class="notice notice-<?php echo esc_attr($issue['type']); ?> inline">
                    <p><?php echo esc_html($issue['message']); ?></p>
                </div>
            <?php endforeach; ?>
        </div>
    <?php endif; ?>
    
    <?php if ($default_cdn_enabled): ?>
        <div class="notice notice-warning inline">
            <p>
                <strong><?php esc_html_e('Notice:', '1cdn-manager'); ?></strong>
                <?php esc_html_e('Site URL Management is disabled because Default CDN (1cdn.us) is currently enabled. To use Site URL Management, please disable Default CDN in', '1cdn-manager'); ?>
                <a href="<?php echo admin_url('admin.php?page=cdn1-settings'); ?>"><?php esc_html_e('CDN Settings', '1cdn-manager'); ?></a>
                <?php esc_html_e('first.', '1cdn-manager'); ?>
            </p>
        </div>
    <?php endif; ?>
    
    <!-- Configuration Form -->
    <div class="cdn1-card" style="margin-top: 20px;">
        <h2><?php esc_html_e('URL Configuration', '1cdn-manager'); ?></h2>
        
        <form method="post" action="">
            <?php 
            wp_nonce_field('cdn1_site_config'); 
            
            // Get auto-detected URLs - do this here to avoid early failures
            $auto_detected = ['site_url' => get_option('home', ''), 'admin_url' => ''];
            if (class_exists('CDN1_Site_Config') && method_exists('CDN1_Site_Config', 'auto_detect_urls')) {
                $detected = CDN1_Site_Config::auto_detect_urls();
                if (is_array($detected)) {
                    $auto_detected = $detected;
                }
            }
            ?>
            
            <table class="form-table">
                <tr>
                    <th scope="row">
                        <label for="enable_site_url_management">
                            <?php esc_html_e('Enable Site URL Management', '1cdn-manager'); ?>
                        </label>
                    </th>
                    <td>
                        <label class="cdn1-switch">
                            <input type="checkbox" 
                                   id="enable_site_url_management" 
                                   name="enable_site_url_management" 
                                   value="1" 
                                   <?php checked($feature_enabled); ?>
                                   <?php disabled($default_cdn_enabled); ?>>
                            <span class="cdn1-slider"></span>
                        </label>
                        <?php if ($default_cdn_enabled): ?>
                            <p class="description" style="color: #d63638;">
                                <?php esc_html_e('Disabled because Default CDN is enabled.', '1cdn-manager'); ?>
                            </p>
                        <?php else: ?>
                            <p class="description">
                                <?php esc_html_e('Enable this to force WordPress to use custom site URLs instead of auto-detected ones.', '1cdn-manager'); ?>
                            </p>
                        <?php endif; ?>
                    </td>
                </tr>
                
                <tr>
                    <th scope="row">
                        <label for="custom_site_url">
                            <?php esc_html_e('Site URL (WP_HOME)', '1cdn-manager'); ?>
                            <span class="required" style="color: #dc3232;">*</span>
                        </label>
                    </th>
                    <td>
                        <input type="url" 
                               id="custom_site_url" 
                               name="custom_site_url" 
                               value="<?php echo esc_attr($custom_site_url); ?>" 
                               class="regular-text cdn1-required-field"
                               placeholder="https://www.mydomain.com"
                               <?php if ($feature_enabled) echo 'required'; ?>
                               <?php disabled(!$feature_enabled); ?>>
                        <p class="description">
                            <?php esc_html_e('The main URL of your website (storefront/public site). This is what visitors see.', '1cdn-manager'); ?>
                            <br>
                            <strong><?php esc_html_e('Auto-detected:', '1cdn-manager'); ?></strong> 
                            <code><?php echo esc_html($auto_detected['site_url']); ?></code>
                            <?php 
                            $frontend_url = get_option('cdn1_cdn_frontend_url', '');
                            $frontend_enabled = get_option('cdn1_enable_frontend_url', 0);
                            if (!empty($frontend_url) && $frontend_enabled): 
                            ?>
                                <span style="color: green;"> (<?php esc_html_e('from CDN Settings → Frontend URL - Enabled', '1cdn-manager'); ?>)</span>
                            <?php elseif (!empty($frontend_url) && !$frontend_enabled): ?>
                                <span style="color: gray;"> (<?php esc_html_e('from CDN Settings → Frontend URL - Disabled', '1cdn-manager'); ?>)</span>
                            <?php endif; ?>
                        </p>
                    </td>
                </tr>
                
                <tr>
                    <th scope="row">
                        <label for="custom_admin_url">
                            <?php esc_html_e('Admin URL (WP_ADMIN_URL)', '1cdn-manager'); ?>
                            <span class="required" style="color: #dc3232;">*</span>
                        </label>
                    </th>
                    <td>
                        <input type="url" 
                               id="custom_admin_url" 
                               name="custom_admin_url" 
                               value="<?php echo esc_attr($custom_admin_url); ?>" 
                               class="regular-text cdn1-required-field"
                               placeholder="https://admin.mydomain.com"
                               <?php if ($feature_enabled) echo 'required'; ?>
                               <?php disabled(!$feature_enabled); ?>>
                        <p class="description">
                            <?php esc_html_e('The URL for accessing WordPress admin panel. Required when Site URL Management is enabled.', '1cdn-manager'); ?>
                            <br>
                            <?php if (!empty($auto_detected['admin_url'])): ?>
                                <strong><?php esc_html_e('Auto-detected:', '1cdn-manager'); ?></strong> 
                                <code><?php echo esc_html($auto_detected['admin_url']); ?></code>
                                <?php 
                                $cdn_admin_url = get_option('cdn1_admin_url', '');
                                $admin_enabled = get_option('cdn1_enable_admin_url', 0);
                                if (!empty($cdn_admin_url) && $admin_enabled): 
                                ?>
                                    <span style="color: green;"> (<?php esc_html_e('from CDN Settings → Admin URL - Enabled', '1cdn-manager'); ?>)</span>
                                <?php elseif (!empty($cdn_admin_url) && !$admin_enabled): ?>
                                    <span style="color: gray;"> (<?php esc_html_e('from CDN Settings → Admin URL - Disabled', '1cdn-manager'); ?>)</span>
                                <?php endif; ?>
                            <?php else: ?>
                                <strong><?php esc_html_e('Example:', '1cdn-manager'); ?></strong> 
                                <code>https://admin.mydomain.com</code>
                            <?php endif; ?>
                        </p>
                    </td>
                </tr>
            </table>
            
            <p class="submit">
                <button type="submit" name="cdn1_save_site_config" class="button button-primary">
                    <?php esc_html_e('Save Configuration', '1cdn-manager'); ?>
                </button>
                <button type="button" id="cdn1-auto-fill-urls" class="button button-secondary">
                    <?php esc_html_e('Auto-Fill Detected URLs', '1cdn-manager'); ?>
                </button>
            </p>
            
            <?php
            // Show helpful tip about CDN Settings
            $frontend_url = CDN1_Settings::get('cdn_frontend_url');
            $origin_url = CDN1_Settings::get('origin_url');
            if (empty($frontend_url) && empty($origin_url)):
            ?>
            <div class="notice notice-info inline">
                <p>
                    <strong><?php esc_html_e('Tip:', '1cdn-manager'); ?></strong>
                    <?php esc_html_e('For better auto-detection, configure your Frontend URL and Admin URL in', '1cdn-manager'); ?>
                    <a href="<?php echo admin_url('admin.php?page=cdn1-settings'); ?>"><?php esc_html_e('CDN Settings', '1cdn-manager'); ?></a>
                    <?php esc_html_e('first. This will help auto-fill the correct URLs.', '1cdn-manager'); ?>
                </p>
            </div>
            <?php endif; ?>
            
            <?php
            // Show helpful message if configured values differ from detected
            $configured_admin = CDN1_Settings::get('custom_admin_url');
            $detected_admin = CDN1_Settings::get('admin_url');
            if ($feature_enabled && !empty($configured_admin) && !empty($detected_admin) && $configured_admin !== rtrim($detected_admin, '/')): 
            ?>
            <div class="notice notice-warning inline">
                <p>
                    <strong><?php esc_html_e('Notice:', '1cdn-manager'); ?></strong>
                    <?php esc_html_e('Your configured Admin URL differs from the Admin URL in CDN Settings. Click "Auto-Fill Detected URLs" to sync with CDN Settings.', '1cdn-manager'); ?>
                </p>
            </div>
            <?php endif; ?>
        </form>
    </div>
    
    <!-- Diagnostics -->
    <div class="cdn1-card" style="margin-top: 20px;">
        <h2>
            <span class="dashicons dashicons-info"></span>
            <?php esc_html_e('System Diagnostics', '1cdn-manager'); ?>
        </h2>
        
        <?php
        // Load diagnostics here, right before use
        $diagnostics = [
            'http_host' => $_SERVER['HTTP_HOST'] ?? 'Unknown',
            'server_name' => $_SERVER['SERVER_NAME'] ?? 'Unknown',
            'is_ssl' => is_ssl(),
            'detected_home_url' => get_option('home', ''),
            'detected_site_url' => get_option('siteurl', ''),
            'wp_home_constant' => defined('WP_HOME') ? WP_HOME : 'Not defined',
            'wp_siteurl_constant' => defined('WP_SITEURL') ? WP_SITEURL : 'Not defined',
            'wp_admin_url_constant' => defined('WP_ADMIN_URL') ? WP_ADMIN_URL : 'Not defined',
            'feature_enabled' => $feature_enabled,
            'custom_site_url' => $custom_site_url,
            'custom_admin_url' => $custom_admin_url,
            'effective_home_url' => home_url(),
            'effective_site_url' => site_url(),
            'admin_url' => admin_url(),
        ];
        
        // Load issues and affected plugins
        $issues = [];
        $affected_plugins = [];
        if (class_exists('CDN1_Site_Config')) {
            if (method_exists('CDN1_Site_Config', 'check_configuration')) {
                $check_result = CDN1_Site_Config::check_configuration();
                if (is_array($check_result)) {
                    $issues = $check_result;
                }
            }
            if (method_exists('CDN1_Site_Config', 'get_affected_plugins')) {
                $plugins_result = CDN1_Site_Config::get_affected_plugins();
                if (is_array($plugins_result)) {
                    $affected_plugins = $plugins_result;
                }
            }
        }
        ?>
        
        <table class="widefat striped">
            <thead>
                <tr>
                    <th><?php esc_html_e('Parameter', '1cdn-manager'); ?></th>
                    <th><?php esc_html_e('Value', '1cdn-manager'); ?></th>
                </tr>
            </thead>
            <tbody>
                <tr>
                    <td><strong><?php esc_html_e('Current HTTP Host', '1cdn-manager'); ?></strong></td>
                    <td><code><?php echo esc_html($diagnostics['http_host']); ?></code></td>
                </tr>
                <tr>
                    <td><strong><?php esc_html_e('Server Name', '1cdn-manager'); ?></strong></td>
                    <td><code><?php echo esc_html($diagnostics['server_name']); ?></code></td>
                </tr>
                <tr>
                    <td><strong><?php esc_html_e('SSL Status', '1cdn-manager'); ?></strong></td>
                    <td>
                        <?php if ($diagnostics['is_ssl']): ?>
                            <span style="color: green;">✓ <?php esc_html_e('Enabled', '1cdn-manager'); ?></span>
                        <?php else: ?>
                            <span style="color: orange;">✗ <?php esc_html_e('Disabled', '1cdn-manager'); ?></span>
                        <?php endif; ?>
                    </td>
                </tr>
                <tr class="alternate">
                    <td><strong><?php esc_html_e('Detected Home URL', '1cdn-manager'); ?></strong></td>
                    <td><code><?php echo esc_html($diagnostics['detected_home_url']); ?></code></td>
                </tr>
                <tr class="alternate">
                    <td><strong><?php esc_html_e('Detected Site URL', '1cdn-manager'); ?></strong></td>
                    <td><code><?php echo esc_html($diagnostics['detected_site_url']); ?></code></td>
                </tr>
                <tr>
                    <td><strong><?php esc_html_e('WP_HOME Constant', '1cdn-manager'); ?></strong></td>
                    <td>
                        <code><?php echo esc_html($diagnostics['wp_home_constant']); ?></code>
                        <?php if ($diagnostics['wp_home_constant'] !== 'Not defined'): ?>
                            <span style="color: orange;"> ⚠ <?php esc_html_e('(Overrides plugin settings)', '1cdn-manager'); ?></span>
                        <?php endif; ?>
                    </td>
                </tr>
                <tr>
                    <td><strong><?php esc_html_e('WP_SITEURL Constant', '1cdn-manager'); ?></strong></td>
                    <td>
                        <code><?php echo esc_html($diagnostics['wp_siteurl_constant']); ?></code>
                        <?php if ($diagnostics['wp_siteurl_constant'] !== 'Not defined'): ?>
                            <span style="color: orange;"> ⚠ <?php esc_html_e('(Overrides plugin settings)', '1cdn-manager'); ?></span>
                        <?php endif; ?>
                    </td>
                </tr>
                <tr>
                    <td><strong><?php esc_html_e('WP_ADMIN_URL Constant', '1cdn-manager'); ?></strong></td>
                    <td>
                        <code><?php echo esc_html($diagnostics['wp_admin_url_constant']); ?></code>
                        <?php if ($diagnostics['wp_admin_url_constant'] !== 'Not defined'): ?>
                            <span style="color: orange;"> ⚠ <?php esc_html_e('(Overrides plugin settings)', '1cdn-manager'); ?></span>
                        <?php endif; ?>
                    </td>
                </tr>
                <tr class="alternate">
                    <td><strong><?php esc_html_e('Feature Status', '1cdn-manager'); ?></strong></td>
                    <td>
                        <?php if ($diagnostics['feature_enabled']): ?>
                            <span style="color: green; font-weight: bold;">✓ <?php esc_html_e('ACTIVE', '1cdn-manager'); ?></span>
                        <?php else: ?>
                            <span style="color: gray;">✗ <?php esc_html_e('Inactive', '1cdn-manager'); ?></span>
                        <?php endif; ?>
                    </td>
                </tr>
                <tr class="alternate">
                    <td><strong><?php esc_html_e('Configured Site URL', '1cdn-manager'); ?></strong></td>
                    <td>
                        <?php if (!empty($diagnostics['custom_site_url'])): ?>
                            <code><?php echo esc_html($diagnostics['custom_site_url']); ?></code>
                        <?php else: ?>
                            <em><?php esc_html_e('Not configured', '1cdn-manager'); ?></em>
                        <?php endif; ?>
                    </td>
                </tr>
                <tr class="alternate">
                    <td><strong><?php esc_html_e('Configured Admin URL', '1cdn-manager'); ?></strong></td>
                    <td>
                        <?php if (!empty($diagnostics['custom_admin_url'])): ?>
                            <code><?php echo esc_html($diagnostics['custom_admin_url']); ?></code>
                        <?php else: ?>
                            <em><?php esc_html_e('Not configured (using Site URL)', '1cdn-manager'); ?></em>
                        <?php endif; ?>
                    </td>
                </tr>
                <tr>
                    <td><strong><?php esc_html_e('Effective Home URL', '1cdn-manager'); ?></strong></td>
                    <td>
                        <code><?php echo esc_html($diagnostics['effective_home_url']); ?></code>
                        <?php if ($feature_enabled && $diagnostics['effective_home_url'] === $diagnostics['custom_site_url']): ?>
                            <span style="color: green;"> ✓ <?php esc_html_e('(Using configured value)', '1cdn-manager'); ?></span>
                        <?php endif; ?>
                    </td>
                </tr>
                <tr>
                    <td><strong><?php esc_html_e('Effective Site URL', '1cdn-manager'); ?></strong></td>
                    <td>
                        <code><?php echo esc_html($diagnostics['effective_site_url']); ?></code>
                        <?php if ($feature_enabled && $diagnostics['effective_site_url'] === $diagnostics['custom_site_url']): ?>
                            <span style="color: green;"> ✓ <?php esc_html_e('(Using configured value)', '1cdn-manager'); ?></span>
                        <?php endif; ?>
                    </td>
                </tr>
                <tr>
                    <td><strong><?php esc_html_e('Admin URL', '1cdn-manager'); ?></strong></td>
                    <td><code><?php echo esc_html($diagnostics['admin_url']); ?></code></td>
                </tr>
            </tbody>
        </table>
        
        <?php
        // Check if WordPress URLs are incorrect (set to admin domain instead of frontend)
        $wp_home = get_option('home');
        $wp_siteurl = get_option('siteurl');
        $current_host = $_SERVER['HTTP_HOST'] ?? '';
        
        // Show fix button if URLs contain admin subdomain
        $needs_fix = (strpos($wp_home, 'admin.') !== false || strpos($wp_siteurl, 'admin.') !== false ||
                     strpos($wp_home, '-admin.') !== false || strpos($wp_siteurl, '-admin.') !== false);
        
        if ($needs_fix): ?>
            <div class="notice notice-error inline" style="margin-top: 15px;">
                <p>
                    <strong><?php esc_html_e('⚠️ WordPress URL Issue Detected!', '1cdn-manager'); ?></strong><br>
                    <?php esc_html_e('Your WordPress Home and Site URLs are set to the admin domain. This causes incorrect URL detection.', '1cdn-manager'); ?><br>
                    <strong><?php esc_html_e('Current:', '1cdn-manager'); ?></strong> <code><?php echo esc_html($wp_home); ?></code><br>
                    <?php esc_html_e('Click below to fix these URLs automatically.', '1cdn-manager'); ?>
                </p>
                <p>
                    <button type="button" id="cdn1-fix-wp-urls" class="button button-primary">
                        <?php esc_html_e('🔧 Fix WordPress URLs Now', '1cdn-manager'); ?>
                    </button>
                    <span id="cdn1-fix-url-result" style="margin-left: 10px;"></span>
                </p>
            </div>
        <?php endif; ?>
    </div>
    
    <!-- Affected Plugins -->
    <?php if (!empty($affected_plugins)): ?>
    <div class="cdn1-card" style="margin-top: 20px;">
        <h2>
            <span class="dashicons dashicons-admin-plugins"></span>
            <?php esc_html_e('Potentially Affected Plugins', '1cdn-manager'); ?>
        </h2>
        <p>
            <?php esc_html_e('The following active plugins are known to be affected by incorrect site URL detection:', '1cdn-manager'); ?>
        </p>
        <ul style="list-style: disc; margin-left: 20px;">
            <?php foreach ($affected_plugins as $plugin): ?>
                <li><strong><?php echo esc_html($plugin); ?></strong></li>
            <?php endforeach; ?>
        </ul>
        <p class="description">
            <?php esc_html_e('After enabling Site URL Management, you may need to reconnect or reconfigure these plugins.', '1cdn-manager'); ?>
        </p>
    </div>
    <?php endif; ?>
    
    <!-- Help Section -->
    <div class="cdn1-card" style="margin-top: 20px;">
        <h2>
            <span class="dashicons dashicons-sos"></span>
            <?php esc_html_e('How It Works', '1cdn-manager'); ?>
        </h2>
        <div class="cdn1-help-content">
            <h3><?php esc_html_e('Problem:', '1cdn-manager'); ?></h3>
            <p>
                <?php esc_html_e('When WordPress admin is accessed from a different URL (e.g., admin.mydomain.com instead of www.mydomain.com), many plugins detect the admin URL as the site URL. This causes issues with:', '1cdn-manager'); ?>
            </p>
            <ul style="list-style: disc; margin-left: 20px;">
                <li><?php esc_html_e('Google Site Kit - requires reconnection', '1cdn-manager'); ?></li>
                <li><?php esc_html_e('Jetpack - connection issues', '1cdn-manager'); ?></li>
                <li><?php esc_html_e('SEO plugins - wrong canonical URLs', '1cdn-manager'); ?></li>
                <li><?php esc_html_e('Analytics plugins - tracking incorrect domain', '1cdn-manager'); ?></li>
            </ul>
            
            <h3><?php esc_html_e('Solution:', '1cdn-manager'); ?></h3>
            <p>
                <?php esc_html_e('This feature forces WordPress to use your configured URLs instead of auto-detected ones. It uses WordPress filters to override home_url() and site_url() functions. WP_SITEURL automatically uses the same value as Site URL.', '1cdn-manager'); ?>
            </p>
            
            <h3><?php esc_html_e('Setup Steps:', '1cdn-manager'); ?></h3>
            <ol style="list-style: decimal; margin-left: 20px;">
                <li><?php esc_html_e('Configure Frontend URL and Admin URL in CDN Settings first (recommended)', '1cdn-manager'); ?></li>
                <li><?php esc_html_e('Feature auto-enables when Admin URL is configured', '1cdn-manager'); ?></li>
                <li><?php esc_html_e('Or manually: Enable "Site URL Management" checkbox', '1cdn-manager'); ?></li>
                <li><?php esc_html_e('Enter your main site URL - e.g., https://www.mydomain.com', '1cdn-manager'); ?></li>
                <li><?php esc_html_e('Optional: Enter your admin URL if different - e.g., https://admin.mydomain.com', '1cdn-manager'); ?></li>
                <li><?php esc_html_e('Click "Save Configuration" or use "Auto-Fill Detected URLs" button', '1cdn-manager'); ?></li>
                <li><?php esc_html_e('Reconnect affected plugins (e.g., Google Site Kit)', '1cdn-manager'); ?></li>
            </ol>
            
            <h3><?php esc_html_e('Note:', '1cdn-manager'); ?></h3>
            <p>
                <?php esc_html_e('If WP_HOME or WP_SITEURL constants are defined in wp-config.php, they will take priority over these settings. Remove those constants from wp-config.php to use plugin-managed URLs. WP_SITEURL automatically mirrors Site URL value.', '1cdn-manager'); ?>
            </p>
        </div>
    </div>
</div>

<script>
jQuery(document).ready(function($) {
    // Auto-fill detected URLs
    $('#cdn1-auto-fill-urls').on('click', function() {
        $('#custom_site_url').val('<?php echo esc_js($auto_detected['site_url']); ?>');
        $('#custom_admin_url').val('<?php echo esc_js($auto_detected['admin_url']); ?>');
    });
    
    // Enable/disable inputs based on feature toggle
    $('#enable_site_url_management').on('change', function() {
        const enabled = $(this).is(':checked');
        $('#custom_site_url, #custom_admin_url').prop('disabled', !enabled);
        
        // Add/remove required attribute based on toggle state
        if (enabled) {
            $('#custom_site_url, #custom_admin_url').prop('required', true);
        } else {
            $('#custom_site_url, #custom_admin_url').prop('required', false);
        }
    });
    
    // Fix WordPress URLs button
    $('#cdn1-fix-wp-urls').on('click', function() {
        const button = $(this);
        const result = $('#cdn1-fix-url-result');
        
        button.prop('disabled', true).text('<?php esc_html_e('Fixing...', '1cdn-manager'); ?>');
        result.html('<span style="color: #666;">⏳ <?php esc_html_e('Processing...', '1cdn-manager'); ?></span>');
        
        $.ajax({
            url: ajaxurl,
            type: 'POST',
            data: {
                action: 'cdn1_fix_wordpress_urls',
                nonce: '<?php echo wp_create_nonce('cdn1_fix_wp_urls'); ?>'
            },
            success: function(response) {
                if (response.success) {
                    result.html('<span style="color: green;">✓ ' + response.data.message + '</span>');
                    setTimeout(function() {
                        location.reload();
                    }, 1500);
                } else {
                    result.html('<span style="color: red;">✗ ' + response.data.message + '</span>');
                    button.prop('disabled', false).text('<?php esc_html_e('🔧 Fix WordPress URLs Now', '1cdn-manager'); ?>');
                }
            },
            error: function() {
                result.html('<span style="color: red;">✗ <?php esc_html_e('Error occurred', '1cdn-manager'); ?></span>');
                button.prop('disabled', false).text('<?php esc_html_e('🔧 Fix WordPress URLs Now', '1cdn-manager'); ?>');
            }
        });
    });
});
</script>

<style>
.cdn1-card {
    background: #fff;
    border: 1px solid #ccd0d4;
    border-radius: 4px;
    padding: 20px;
    box-shadow: 0 1px 1px rgba(0,0,0,.04);
}

.cdn1-card h2 {
    margin-top: 0;
    padding-bottom: 10px;
    border-bottom: 1px solid #eee;
}

.cdn1-diagnostics-issues {
    margin: 20px 0;
}

.cdn1-help-content h3 {
    margin-top: 20px;
    margin-bottom: 10px;
    color: #2271b1;
}

.cdn1-help-content ul,
.cdn1-help-content ol {
    margin-bottom: 15px;
}

.required {
    font-weight: bold;
}

/* Fix column width ratio - reduce left column to give more space to right */
.form-table th {
    width: 180px;
    max-width: 180px;
    vertical-align: top;
}

.form-table td {
    width: auto;
}
</style>
<?php
