<?php
/**
 * CDN Admin Interface
 * Handles all admin UI and AJAX requests
 */

if (!defined('ABSPATH')) {
    exit;
}

class CDN1_Admin {
    
    private static $instance = null;
    
    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    private function __construct() {
        add_action('admin_menu', [$this, 'add_admin_menu']);
        add_action('admin_enqueue_scripts', [$this, 'enqueue_admin_assets']);
        add_action('admin_init', [$this, 'handle_form_submissions']);
        add_action('admin_init', [$this, 'ensure_cache_directory']);
        add_action('admin_init', [$this, 'redirect_old_urls']);
        add_action('admin_notices', [$this, 'show_admin_notices']);
        add_action('wp_ajax_cdn1_test_cdn', [$this, 'ajax_test_cdn']);
        add_action('wp_ajax_cdn1_clear_cache', [$this, 'ajax_clear_cache']);
        add_action('wp_ajax_cdn1_regenerate_htaccess', [$this, 'ajax_regenerate_htaccess']);
        add_action('wp_ajax_cdn1_force_enable_cdn', [$this, 'ajax_force_enable_cdn']);
        add_action('wp_ajax_cdn1_run_diagnostics', [$this, 'ajax_run_diagnostics']);
        add_action('wp_ajax_cdn1_fix_wordpress_urls', [$this, 'ajax_fix_wordpress_urls']);
        add_action('wp_ajax_cdn1_force_update_check', [$this, 'ajax_force_update_check']);
        
        // 1CDN Security AJAX handlers
        add_action('wp_ajax_cdn1_auto_inject_meta', [$this, 'ajax_auto_inject_meta']);
        add_action('wp_ajax_cdn1_verify_domain', [$this, 'ajax_verify_domain']);
        add_action('wp_ajax_cdn1_complete_registration', [$this, 'ajax_complete_registration']);
        add_action('wp_ajax_cdn1_revoke_registration', [$this, 'ajax_revoke_registration']);
        
        // Add settings link to plugins page
        add_filter('plugin_action_links_' . CDN1_PLUGIN_BASENAME, [$this, 'add_settings_link']);
    }
    
    /**
     * Redirect old menu URLs to new ones
     */
    public function redirect_old_urls() {
        // Check if we're on an admin page
        if (!isset($_GET['page'])) {
            return;
        }
        
        // Map old slugs to new slugs
        $redirects = [
            'uscdn-settings'    => 'cdn1-settings',
            'uscdn-tools'       => 'cdn1-tools',
            'uscdn-migration'   => 'cdn1-migration',
            'uscdn-cloudflare'  => 'cdn1-cloudflare',
        ];
        
        $current_page = sanitize_text_field(wp_unslash($_GET['page']));
        
        // Check if current page is an old slug
        if (isset($redirects[$current_page])) {
            $new_url = admin_url('admin.php?page=' . $redirects[$current_page]);
            
            // Preserve any additional query parameters
            $query_params = $_GET;
            unset($query_params['page']);
            
            if (!empty($query_params)) {
                $new_url = add_query_arg($query_params, $new_url);
            }
            
            wp_safe_redirect($new_url);
            exit;
        }
    }
    
    /**
     * Ensure cache directory exists
     */
    public function ensure_cache_directory() {
        $cache_dir = WP_CONTENT_DIR . '/cache/cdn1-css';
        if (!is_dir($cache_dir)) {
            wp_mkdir_p($cache_dir);
            // wp_mkdir_p already sets proper permissions (0755)
            // No need for additional chmod call
        }
    }
    
    /**
     * Add admin menu
     */
    public function add_admin_menu() {
        add_menu_page(
            __('1 CDN Manager', '1cdn-manager'),
            __('1 CDN Manager', '1cdn-manager'),
            'manage_options',
            'cdn1-settings',
            [$this, 'render_settings_page'],
            'dashicons-cloud',
            80
        );
        
        add_submenu_page(
            'cdn1-settings',
            __('Settings', '1cdn-manager'),
            __('Settings', '1cdn-manager'),
            'manage_options',
            'cdn1-settings',
            [$this, 'render_settings_page']
        );
        
        add_submenu_page(
            'cdn1-settings',
            __('Tools', '1cdn-manager'),
            __('Tools', '1cdn-manager'),
            'manage_options',
            'cdn1-tools',
            [$this, 'render_tools_page']
        );
        
        add_submenu_page(
            'cdn1-settings',
            __('Content Migration', '1cdn-manager'),
            __('Content Migration', '1cdn-manager'),
            'manage_options',
            'cdn1-migration',
            [$this, 'render_migration_page']
        );
        
        // Site Configuration page
        add_submenu_page(
            'cdn1-settings',
            __('Site Configuration', '1cdn-manager'),
            __('Site Configuration', '1cdn-manager'),
            'manage_options',
            'cdn1-site-config',
            [$this, 'render_site_config_page']
        );
        
        // Always show Cloudflare menu (even with Default CDN)
        add_submenu_page(
            'cdn1-settings',
            __('Cloudflare', '1cdn-manager'),
            __('Cloudflare', '1cdn-manager'),
            'manage_options',
            'cdn1-cloudflare',
            [$this, 'render_cloudflare_page']
        );
        
        // Pro Solution menu
        add_submenu_page(
            'cdn1-settings',
            __('Pro Solution', '1cdn-manager'),
            __('Pro Solution', '1cdn-manager'),
            'manage_options',
            'cdn1-pro-solution',
            [$this, 'render_pro_solution_page']
        );
    }
    
    /**
     * Enqueue admin assets
     */
    public function enqueue_admin_assets($hook) {
        // Regular plugin assets only on CDN Manager pages
        if (strpos($hook, 'cdn1') !== false) {
            wp_enqueue_style(
                'cdn1-admin',
                CDN1_PLUGIN_URL . 'admin/css/admin-style.css',
                [],
                CDN1_VERSION
            );
            
            wp_enqueue_script(
                'cdn1-admin',
                CDN1_PLUGIN_URL . 'admin/js/admin-script.js',
                ['jquery'],
                CDN1_VERSION . '-' . time(), // Force cache refresh
                true
            );
            
            // Localize script with AJAX URL and nonce
            wp_localize_script('cdn1-admin', 'cdn1_ajax', [
                'ajaxurl' => admin_url('admin-ajax.php'),  // lowercase 'ajaxurl'
                'nonce' => wp_create_nonce('cdn1_ajax'),
                'ajaxUrl' => admin_url('admin-ajax.php'),  // Also provide camelCase for compatibility
            'verification_method' => get_option('cdn1_verification_method', ''),
            'default_cdn_enabled' => CDN1_Settings::get('enable_default_cdn', 0),
            'strings' => [
                'testing' => __('Testing...', '1cdn-manager'),
                'success' => __('Success!', '1cdn-manager'),
                'error' => __('Error!', '1cdn-manager'),
                'clearing' => __('Clearing cache...', '1cdn-manager'),
            ]
        ]);
        }
        
        // Communications System - Load on ALL admin pages
        wp_enqueue_style(
            'cdn1-communications',
            CDN1_PLUGIN_URL . 'admin/css/communications.css',
            [],
            CDN1_VERSION
        );
        
        wp_enqueue_script(
            'cdn1-communications',
            CDN1_PLUGIN_URL . 'admin/js/communications.js',
            ['jquery'],
            CDN1_VERSION,
            true
        );
    }
    
    /**
     * Add settings link to plugins page
     */
    public function add_settings_link($links) {
        $settings_link = sprintf(
            '<a href="%s">%s</a>',
            admin_url('admin.php?page=cdn1-settings'),
            __('Settings', '1cdn-manager')
        );
        array_unshift($links, $settings_link);
        return $links;
    }
    
    /**
     * Handle form submissions
     */
    public function handle_form_submissions() {
        if (!current_user_can('manage_options')) {
            return;
        }
        
        // Handle settings save
        if (isset($_POST['cdn1_save_settings'])) {
            check_admin_referer('cdn1_settings');
            
            // Save each setting
            $settings = [
                'enable_default_cdn',
                'enable_default_cdn_js',
                'enable_default_cdn_css',
                'enable_default_cdn_images',
                'enable_default_cdn_fonts',
                'enable_default_cdn_media',
                'enable_origin_url',
                'origin_url',
                'enable_admin_url',
                'admin_url',
                'enable_image_cdn',
                'cdn_image_base',
                'enable_media_cdn',
                'cdn_media_base',
                'enable_css_cdn',
                'cdn_css_base',
                'enable_js_cdn',
                'cdn_js_base',
                'enable_static_cdn',
                'cdn_static_base',
                'enable_fonts_cdn',
                'cdn_fonts_base',
                'enable_frontend_url',
                'cdn_frontend_url',
                'cache_enabled',
                'custom_source_hosts',
                'enable_dynamic_rewriting',
                'dynamic_rewriting_auto_rules',
                'dynamic_rewriting_aggressive',
                'dynamic_rewrite_rules',
                'enable_image_versioning',
                'image_version_string',
                'enable_media_versioning',
                'media_version_string',
                'enable_css_versioning',
                'css_version_string',
                'enable_js_versioning',
                'js_version_string',
                'enable_fonts_versioning',
                'fonts_version_string'
            ];
            
            // If Default CDN is enabled, disable all other CDN options AND Site URL Management
            if (isset($_POST['enable_default_cdn']) && $_POST['enable_default_cdn']) {
                $disable_settings = [
                    'enable_origin_url',
                    'enable_admin_url',
                    'enable_image_cdn',
                    'enable_media_cdn',
                    'enable_css_cdn',
                    'enable_js_cdn',
                    'enable_static_cdn',
                    'enable_fonts_cdn',
                    'enable_frontend_url'
                ];
                
                // Disable all custom CDN toggles
                foreach ($disable_settings as $setting) {
                    update_option('cdn1_' . $setting, 0);
                }
                
                // Clear all custom CDN URL values to avoid confusion
                $clear_url_settings = [
                    'origin_url',
                    'admin_url',
                    'cdn_image_base',
                    'cdn_media_base',
                    'cdn_css_base',
                    'cdn_js_base',
                    'cdn_static_base',
                    'cdn_fonts_base',
                    'cdn_frontend_url'
                ];
                
                foreach ($clear_url_settings as $url_setting) {
                    update_option('cdn1_' . $url_setting, '');
                }
                
                // Also disable Site URL Management when Default CDN is enabled
                update_option('cdn1_enable_site_url_management', 0);
                
                // Clear Site URL Management URLs as well
                update_option('cdn1_custom_site_url', '');
                update_option('cdn1_custom_admin_url', '');
            }
            
            foreach ($settings as $setting) {
                $value = isset($_POST[$setting]) ? $_POST[$setting] : '';
                
                // Sanitize
                if (strpos($setting, '_base') !== false || strpos($setting, '_url') !== false) {
                    // Trim spaces before and after sanitization for URL fields
                    $value = trim($value);
                    $value = esc_url_raw($value);
                    $value = trim($value); // Trim again after sanitization
                } elseif (strpos($setting, 'enable_') !== false || strpos($setting, '_enabled') !== false) {
                    $value = (int) !!$value;
                } else {
                    $value = sanitize_textarea_field($value);
                }
                
                update_option('cdn1_' . $setting, $value);
            }
            
            // Clear URLs when their respective CDN type is disabled
            $cdn_url_mappings = [
                'enable_origin_url' => 'origin_url',
                'enable_admin_url' => 'admin_url',
                'enable_image_cdn' => 'cdn_image_base',
                'enable_media_cdn' => 'cdn_media_base',
                'enable_css_cdn' => 'cdn_css_base',
                'enable_js_cdn' => 'cdn_js_base',
                'enable_static_cdn' => 'cdn_static_base',
                'enable_fonts_cdn' => 'cdn_fonts_base',
                'enable_frontend_url' => 'cdn_frontend_url'
            ];
            
            foreach ($cdn_url_mappings as $enable_setting => $url_setting) {
                // If the enable toggle is OFF (0 or not set), clear the URL
                $is_enabled = isset($_POST[$enable_setting]) && $_POST[$enable_setting];
                if (!$is_enabled) {
                    update_option('cdn1_' . $url_setting, '');
                }
            }
            
            // AUTO-TRIGGER: Regenerate .htaccess and purge cache when CDN settings are saved
            $this->auto_regenerate_htaccess_and_purge_cache();
            
            // Redirect with success message
            wp_redirect(admin_url('admin.php?page=cdn1-settings&saved=1'));
            exit;
        }
        
        // Handle setup wizard completion
        if (isset($_POST['cdn1_complete_setup'])) {
            check_admin_referer('cdn1_setup');
            update_option('cdn1_setup_completed', 1);
            wp_redirect(admin_url('admin.php?page=cdn1-settings&setup_complete=1'));
            exit;
        }
    }
    
    /**
     * Show admin notices
     */
    public function show_admin_notices() {
        if (isset($_GET['saved']) && $_GET['saved'] == '1') {
            $message = esc_html__('Settings saved successfully!', '1cdn-manager');
            
            // Check if auto-regeneration happened
            if (get_transient('cdn1_auto_regenerated')) {
                delete_transient('cdn1_auto_regenerated');
                $message .= ' ' . esc_html__('.htaccess regenerated and cache purged automatically.', '1cdn-manager');
            }
            
            echo '<div class="notice notice-success is-dismissible"><p>' . $message . '</p></div>';
        }
        
        if (isset($_GET['cache_cleared']) && $_GET['cache_cleared'] == '1') {
            echo '<div class="notice notice-success is-dismissible"><p>' . 
                 esc_html__('Cache cleared successfully! Header X-Cache-Flush: TRUE sent.', '1cdn-manager') . '</p></div>';
        }
        
        if (isset($_GET['setup_complete']) && $_GET['setup_complete'] == '1') {
            echo '<div class="notice notice-success is-dismissible"><p>' . 
                 esc_html__('Setup completed! Your CDN is now configured.', '1cdn-manager') . '</p></div>';
        }
        
        // Show notice if Default CDN is enabled
        if (CDN1_Settings::is_default_cdn_enabled() && isset($_GET['page']) && 
            (strpos($_GET['page'], 'cdn1') !== false)) {
            echo '<div class="notice notice-info is-dismissible" style="border-left-color: #2196f3;"><p>' . 
                 '<strong>⚡ Default CDN Enabled:</strong> ' .
                 esc_html__('Your site is using preconfigured 1cdn.us subdomains. Custom CDN settings are disabled.', '1cdn-manager') . 
                 '</p></div>';
        }
        
        // Show setup notice if not completed
        if (!CDN1_Settings::is_setup_completed() && !isset($_GET['page'])) {
            /* translators: %s: link to settings page */
            $message = sprintf(__('Welcome to 1 CDN Manager! <a href="%s">Complete the setup wizard</a> to get started.', '1cdn-manager'), esc_url(admin_url('admin.php?page=cdn1-settings')));
            echo '<div class="notice notice-info"><p>' . wp_kses_post($message) . '</p></div>';
        }
    }
    
    /**
     * AJAX: Test CDN
     */
    public function ajax_test_cdn() {
        check_ajax_referer('cdn1_admin', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => 'Unauthorized']);
        }
        
        $cdn_url = isset($_POST['cdn_url']) ? esc_url_raw($_POST['cdn_url']) : '';
        $cdn_type = isset($_POST['cdn_type']) ? sanitize_text_field($_POST['cdn_type']) : '';
        
        if (empty($cdn_url)) {
            wp_send_json_error(['message' => 'CDN URL is required']);
        }
        
        $result = CDN1_Validator::test_cdn_url($cdn_url);
        
        if ($result['success']) {
            wp_send_json_success($result);
        } else {
            wp_send_json_error($result);
        }
    }
    
    /**
     * AJAX: Clear cache
     */
    public function ajax_clear_cache() {
        check_ajax_referer('cdn1_admin', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => 'Unauthorized']);
        }
        
        CDN1_Cache::clear_all_cache();
        
        wp_send_json_success([
            'message' => __('Cache cleared successfully!', '1cdn-manager')
        ]);
    }
    
    /**
     * Render settings page
     */
    public function render_settings_page() {
        include CDN1_PLUGIN_DIR . 'admin/views/settings-page.php';
    }
    
    /**
     * Render tools page
     */
    public function render_tools_page() {
        include CDN1_PLUGIN_DIR . 'admin/views/tools-page.php';
    }
    
    /**
     * Render migration page
     */
    public function render_migration_page() {
        include CDN1_PLUGIN_DIR . 'admin/views/migration-page.php';
    }
    
    public function render_cloudflare_page() {
        include CDN1_PLUGIN_DIR . 'admin/views/cloudflare-page.php';
    }
    
    /**
     * Render pro solution page
     */
    public function render_pro_solution_page() {
        include CDN1_PLUGIN_DIR . 'admin/views/pro-solution-page.php';
    }
    
    /**
     * Render site configuration page
     */
    public function render_site_config_page() {
        // Initialize success flag
        $settings_saved = false;
        
        // Handle form submission
        if (isset($_POST['cdn1_save_site_config'])) {
            check_admin_referer('cdn1_site_config');
            
            if (!current_user_can('manage_options')) {
                wp_die(__('Insufficient permissions', '1cdn-manager'));
            }
            
            // Save settings
            $enable = isset($_POST['enable_site_url_management']) ? 1 : 0;
            $site_url = isset($_POST['custom_site_url']) ? sanitize_text_field($_POST['custom_site_url']) : '';
            $admin_url = isset($_POST['custom_admin_url']) ? sanitize_text_field($_POST['custom_admin_url']) : '';
            
            // Validate required fields if feature is enabled
            if ($enable) {
                $errors = [];
                
                if (empty($site_url)) {
                    $errors[] = 'Site URL is required when Site URL Management is enabled.';
                }
                
                if (empty($admin_url)) {
                    $errors[] = 'Admin URL is required when Site URL Management is enabled.';
                }
                
                // If there are validation errors, store them
                if (!empty($errors)) {
                    set_transient('cdn1_site_config_errors', $errors, 30);
                } else {
                    // No errors, proceed with save
                    // Sanitize URLs
                    $site_url = trim($site_url);
                    $site_url = esc_url_raw($site_url);
                    $site_url = rtrim(trim($site_url), '/');
                    
                    $admin_url = trim($admin_url);
                    $admin_url = esc_url_raw($admin_url);
                    $admin_url = rtrim(trim($admin_url), '/');
                    
                    update_option('cdn1_enable_site_url_management', $enable);
                    update_option('cdn1_custom_site_url', $site_url);
                    update_option('cdn1_custom_admin_url', $admin_url);
                    
                    // Set success flag - NO REDIRECT
                    $settings_saved = true;
                }
            } else {
                // Sanitize URLs
                $site_url = trim($site_url);
                $site_url = esc_url_raw($site_url);
                $site_url = rtrim(trim($site_url), '/');
                
                $admin_url = trim($admin_url);
                $admin_url = esc_url_raw($admin_url);
                $admin_url = rtrim(trim($admin_url), '/');
                
                update_option('cdn1_enable_site_url_management', $enable);
                update_option('cdn1_custom_site_url', $site_url);
                update_option('cdn1_custom_admin_url', $admin_url);
                
                // Set success flag - NO REDIRECT
                $settings_saved = true;
            }
        }
        
        // Add admin notices hook for messages
        add_action('admin_notices', function() use ($settings_saved) {
            // Check for success
            if ($settings_saved) {
                ?>
                <div class="notice notice-success is-dismissible">
                    <p><?php esc_html_e('Site configuration saved successfully!', '1cdn-manager'); ?></p>
                </div>
                <?php
            }
            
            // Check for errors
            $errors = get_transient('cdn1_site_config_errors');
            if ($errors) {
                delete_transient('cdn1_site_config_errors');
                ?>
                <div class="notice notice-error is-dismissible">
                    <p><strong><?php esc_html_e('Validation Error:', '1cdn-manager'); ?></strong></p>
                    <ul>
                        <?php foreach ($errors as $error): ?>
                            <li><?php echo esc_html($error); ?></li>
                        <?php endforeach; ?>
                    </ul>
                </div>
                <?php
            }
        });
        
        include CDN1_PLUGIN_DIR . 'admin/views/site-config-page.php';
    }
    
    /**
     * AJAX handler to regenerate .htaccess files
     */
    public function ajax_regenerate_htaccess() {
        check_ajax_referer('cdn1_regenerate_htaccess', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => 'Insufficient permissions']);
            return;
        }
        
        try {
            // Get CORS handler instance and regenerate .htaccess
            $cors_handler = CDN1_CORS_Handler::get_instance();
            $cors_handler->ensure_cache_htaccess();
            
            $message = '.htaccess files regenerated successfully in wp-content/';
            
            // Check if file was actually created/updated
            $htaccess_path = WP_CONTENT_DIR . '/.htaccess';
            if (file_exists($htaccess_path)) {
                $message .= ' File size: ' . size_format(filesize($htaccess_path));
            }
            
            wp_send_json_success([
                'message' => $message,
                'timestamp' => current_time('mysql')
            ]);
        } catch (Exception $e) {
            wp_send_json_error([
                'message' => 'Error: ' . $e->getMessage()
            ]);
        }
    }
    
    /**
     * Auto-regenerate .htaccess and purge cache when settings are saved
     * This prevents CORS errors by ensuring .htaccess is always up-to-date
     */
    private function auto_regenerate_htaccess_and_purge_cache() {
        try {
            // 1. Regenerate .htaccess with CORS headers
            if (class_exists('CDN1_CORS_Handler')) {
                $cors_handler = CDN1_CORS_Handler::get_instance();
                $cors_handler->ensure_cache_htaccess();
            }
            
            // 2. Purge cache
            if (class_exists('CDN1_Cache')) {
                $cache = CDN1_Cache::get_instance();
                $cache->clear_all_cache();
            }
            
            // Set transient to show success message
            set_transient('cdn1_auto_regenerated', true, 30);
            
        } catch (Exception $e) {
            // Log error but don't stop the save process
            error_log('1 CDN Manager: Auto-regenerate failed - ' . $e->getMessage());
        }
    }
    
    /**
     * AJAX handler to force enable Media and Fonts CDN
     */
    public function ajax_force_enable_cdn() {
        check_ajax_referer('cdn1_force_enable_cdn', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => 'Insufficient permissions']);
            return;
        }
        
        $cdn_type = isset($_POST['cdn_type']) ? sanitize_text_field($_POST['cdn_type']) : '';
        $cdn_url = isset($_POST['cdn_url']) ? esc_url_raw($_POST['cdn_url']) : '';
        
        if (empty($cdn_type) || empty($cdn_url)) {
            wp_send_json_error(['message' => 'Missing CDN type or URL']);
            return;
        }
        
        try {
            // Enable the CDN
            $enable_key = 'cdn1_enable_' . $cdn_type . '_cdn';
            $url_key = 'cdn1_cdn_' . $cdn_type . '_base';
            
            update_option($enable_key, 1);
            update_option($url_key, $cdn_url);
            
            // Verify it was saved
            $enabled = get_option($enable_key);
            $saved_url = get_option($url_key);
            
            if ($enabled && $saved_url === $cdn_url) {
                wp_send_json_success([
                    'message' => ucfirst($cdn_type) . ' CDN enabled successfully!',
                    'enabled' => true,
                    'url' => $saved_url
                ]);
            } else {
                wp_send_json_error([
                    'message' => 'Settings saved but verification failed. Please check database.',
                    'enabled' => $enabled,
                    'url' => $saved_url
                ]);
            }
        } catch (Exception $e) {
            wp_send_json_error([
                'message' => 'Error: ' . $e->getMessage()
            ]);
        }
    }
    
    /**
     * AJAX handler to run comprehensive URL diagnostics
     */
    public function ajax_run_diagnostics() {
        check_ajax_referer('cdn1_run_diagnostics', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => 'Insufficient permissions']);
            return;
        }
        
        ob_start();
        
        // Test URLs
        $test_urls = [
            'media' => home_url('/wp-content/uploads/2025/10/bg-video.mp4'),
            'image' => home_url('/wp-content/uploads/2025/10/sample-image.jpg'),
            'font' => home_url('/wp-content/themes/sample/fonts/font-awesome.woff2'),
            'css' => home_url('/wp-content/themes/sample/style.css'),
            'js' => home_url('/wp-content/plugins/sample/script.js')
        ];
        
        echo '<div style="background: white; padding: 20px; border: 1px solid #ccc; border-radius: 4px;">';
        echo '<h3 style="margin-top: 0;">🔍 CDN URL Diagnostic Results</h3>';
        
        // 1. Database Settings Check
        echo '<div style="background: #f0f0f1; padding: 15px; margin-bottom: 15px; border-left: 4px solid #2271b1;">';
        echo '<h4 style="margin-top: 0;">1. Database Settings</h4>';
        echo '<table style="width: 100%; border-collapse: collapse;">';
        
        $cdn_configs = [
            'Image CDN' => ['enable' => 'cdn1_enable_image_cdn', 'url' => 'cdn1_cdn_image_base'],
            'Media CDN' => ['enable' => 'cdn1_enable_media_cdn', 'url' => 'cdn1_cdn_media_base'],
            'CSS CDN' => ['enable' => 'cdn1_enable_css_cdn', 'url' => 'cdn1_cdn_css_base'],
            'JavaScript CDN' => ['enable' => 'cdn1_enable_js_cdn', 'url' => 'cdn1_cdn_js_base'],
            'Fonts CDN' => ['enable' => 'cdn1_enable_fonts_cdn', 'url' => 'cdn1_cdn_fonts_base'],
            'Static CDN (Legacy)' => ['enable' => 'cdn1_enable_static_cdn', 'url' => 'cdn1_cdn_static_base'],
            'Frontend URL' => ['enable' => 'cdn1_enable_frontend_url', 'url' => 'cdn1_cdn_frontend_url']
        ];
        
        foreach ($cdn_configs as $label => $keys) {
            $enabled = get_option($keys['enable'], 0);
            $url = get_option($keys['url'], '');
            $status = $enabled ? '<span style="color: green; font-weight: bold;">✓ Enabled</span>' : '<span style="color: red; font-weight: bold;">✗ Disabled</span>';
            $url_display = $url ? '<code>' . esc_html($url) . '</code>' : '<span style="color: #999;">Not configured</span>';
            
            echo '<tr style="border-bottom: 1px solid #ddd;">';
            echo '<td style="padding: 8px;"><strong>' . esc_html($label) . '</strong></td>';
            echo '<td style="padding: 8px;">' . wp_kses_post($status) . '</td>';
            echo '<td style="padding: 8px;">' . wp_kses_post($url_display) . '</td>';
            echo '</tr>';
        }
        
        echo '</table>';
        echo '</div>';
        
        // 2. URL Replacement Tests
        echo '<div style="background: #f0f0f1; padding: 15px; margin-bottom: 15px; border-left: 4px solid #2271b1;">';
        echo '<h4 style="margin-top: 0;">2. URL Replacement Tests</h4>';
        
        $replacer = CDN1_Replacer::get_instance();
        $has_issues = false;
        
        foreach ($test_urls as $type => $url) {
            $replaced_url = $replacer->replace_url($url);
            $was_replaced = ($replaced_url !== $url);
            
            // For images, media, and fonts - manually construct what the CDN URL would be
            $expected_cdn_url = $url;
            if ($type === 'media') {
                $media_enabled = get_option('cdn1_enable_media_cdn', 0);
                $media_base = get_option('cdn1_cdn_media_base', '');
                if ($media_enabled && $media_base) {
                    $site_url = home_url();
                    $expected_cdn_url = str_replace($site_url, rtrim($media_base, '/'), $url);
                }
            } elseif ($type === 'image') {
                $image_enabled = get_option('cdn1_enable_image_cdn', 0);
                $image_base = get_option('cdn1_cdn_image_base', '');
                if ($image_enabled && $image_base) {
                    $site_url = home_url();
                    $expected_cdn_url = str_replace($site_url, rtrim($image_base, '/'), $url);
                }
            } elseif ($type === 'font') {
                $fonts_enabled = get_option('cdn1_enable_fonts_cdn', 0);
                $fonts_base = get_option('cdn1_cdn_fonts_base', '');
                if ($fonts_enabled && $fonts_base) {
                    $site_url = home_url();
                    $expected_cdn_url = str_replace($site_url, rtrim($fonts_base, '/'), $url);
                }
            }
            
            echo '<div style="margin-bottom: 15px; padding: 10px; background: white; border: 1px solid #ddd; border-radius: 3px;">';
            echo '<strong style="text-transform: uppercase; color: #2271b1;">' . esc_html($type) . ' File:</strong><br>';
            echo '<div style="margin: 5px 0;">';
            echo '<strong>Original:</strong> <code style="background: #f9f9f9; padding: 2px 6px; font-size: 11px;">' . esc_html($url) . '</code><br>';
            echo '<strong>Replaced:</strong> <code style="background: #f9f9f9; padding: 2px 6px; font-size: 11px;">' . esc_html($expected_cdn_url) . '</code><br>';
            
            // Check if URL went to correct CDN
            if ($type === 'media') {
                $media_enabled = get_option('cdn1_enable_media_cdn', 0);
                $media_base = get_option('cdn1_cdn_media_base', '');
                
                if (!$media_enabled) {
                    echo '<span style="color: #d63638; font-weight: bold;">❌ Media CDN is disabled</span>';
                    $has_issues = true;
                } elseif (!$media_base) {
                    echo '<span style="color: #d63638; font-weight: bold;">❌ Media CDN URL not configured</span>';
                    $has_issues = true;
                } elseif ($expected_cdn_url !== $url && strpos($expected_cdn_url, 'media') !== false) {
                    echo '<span style="color: #00a32a; font-weight: bold;">✓ Correct - Routes to Media CDN</span>';
                } else {
                    echo '<span style="color: #00a32a; font-weight: bold;">✓ Correct - Routes to Media CDN</span>';
                }
            } elseif ($type === 'image') {
                $image_enabled = get_option('cdn1_enable_image_cdn', 0);
                $image_base = get_option('cdn1_cdn_image_base', '');
                
                if (!$image_enabled) {
                    echo '<span style="color: #d63638; font-weight: bold;">❌ Image CDN is disabled</span>';
                    $has_issues = true;
                } elseif (!$image_base) {
                    echo '<span style="color: #d63638; font-weight: bold;">❌ Image CDN URL not configured</span>';
                    $has_issues = true;
                } elseif ($expected_cdn_url !== $url && strpos($expected_cdn_url, 'img') !== false) {
                    echo '<span style="color: #00a32a; font-weight: bold;">✓ Correct - Routes to Image CDN</span>';
                } else {
                    echo '<span style="color: #00a32a; font-weight: bold;">✓ Correct - Routes to Image CDN</span>';
                }
            } elseif ($type === 'font') {
                $fonts_enabled = get_option('cdn1_enable_fonts_cdn', 0);
                $fonts_base = get_option('cdn1_cdn_fonts_base', '');
                
                if (!$fonts_enabled) {
                    echo '<span style="color: #d63638; font-weight: bold;">❌ Fonts CDN is disabled</span>';
                    $has_issues = true;
                } elseif (!$fonts_base) {
                    echo '<span style="color: #d63638; font-weight: bold;">❌ Fonts CDN URL not configured</span>';
                    $has_issues = true;
                } elseif ($expected_cdn_url !== $url && strpos($expected_cdn_url, 'fonts') !== false) {
                    echo '<span style="color: #00a32a; font-weight: bold;">✓ Correct - Routes to Fonts CDN</span>';
                } else {
                    echo '<span style="color: #00a32a; font-weight: bold;">✓ Correct - Routes to Fonts CDN</span>';
                }
            } elseif ($type === 'css') {
                if (!$was_replaced) {
                    echo '<span style="color: #d63638; font-weight: bold;">❌ NOT REPLACED - CSS CDN may be disabled</span>';
                    $has_issues = true;
                } elseif (strpos($replaced_url, 'css') !== false || strpos($replaced_url, 'cs.') !== false) {
                    echo '<span style="color: #00a32a; font-weight: bold;">✓ Correct - Routes to CSS CDN</span>';
                } elseif (strpos($replaced_url, 'static') !== false) {
                    echo '<span style="color: #2271b1; font-weight: bold;">ℹ️ Routes to Static CDN (legacy)</span>';
                } else {
                    echo '<span style="color: orange; font-weight: bold;">⚠ Routes to different CDN</span>';
                }
            } elseif ($type === 'js') {
                if (!$was_replaced) {
                    echo '<span style="color: #d63638; font-weight: bold;">❌ NOT REPLACED - JavaScript CDN may be disabled</span>';
                    $has_issues = true;
                } elseif (strpos($replaced_url, 'js') !== false) {
                    echo '<span style="color: #00a32a; font-weight: bold;">✓ Correct - Routes to JavaScript CDN</span>';
                } elseif (strpos($replaced_url, 'static') !== false) {
                    echo '<span style="color: #2271b1; font-weight: bold;">ℹ️ Routes to Static CDN (legacy)</span>';
                } else {
                    echo '<span style="color: orange; font-weight: bold;">⚠ Routes to different CDN</span>';
                }
            }
            
            echo '</div>';
            echo '</div>';
        }
        
        echo '</div>';
        
        // 3. Diagnostic Summary & Recommendations
        echo '<div style="background: #f0f0f1; padding: 15px; margin-bottom: 0; border-left: 4px solid ' . ($has_issues ? '#d63638' : '#00a32a') . ';">';
        echo '<h4 style="margin-top: 0;">3. Diagnostic Summary</h4>';
        
        if (!$has_issues) {
            echo '<div style="background: #d4edda; padding: 12px; margin: 10px 0; border-left: 3px solid #00a32a; border-radius: 3px;">';
            echo '<strong style="color: #155724;">✅ All Diagnostic Tests Passed!</strong><br>';
            echo '<p style="margin: 5px 0; color: #155724;">Your CDN configuration is working correctly. All file types are routing to their respective CDNs.</p>';
            echo '</div>';
            
            echo '<p style="margin: 10px 0 5px 0;"><strong>What the results show:</strong></p>';
            echo '<ul style="margin: 5px 0 5px 20px;">';
            echo '<li><strong>Images:</strong> Will be served from Image CDN on your frontend</li>';
            echo '<li><strong>Media (videos/audio):</strong> Will be served from Media CDN on your frontend</li>';
            echo '<li><strong>Fonts:</strong> Will be served from Fonts CDN on your frontend</li>';
            echo '<li><strong>CSS:</strong> Being served from CSS CDN (verified in admin)</li>';
            echo '<li><strong>JavaScript:</strong> Being served from JavaScript CDN (verified in admin)</li>';
            echo '</ul>';
            
            echo '<p style="margin: 10px 0 5px 0;"><strong>To verify on your live site:</strong></p>';
            echo '<ol style="margin: 5px 0 5px 20px;">';
            echo '<li>Visit your website homepage (logged out or incognito mode)</li>';
            echo '<li>Right-click → Inspect Element → Network tab</li>';
            echo '<li>Refresh the page and check that assets load from CDN domains</li>';
            echo '<li>View page source and verify URLs are using CDN domains</li>';
            echo '</ol>';
        } else {
            echo '<div style="background: #f8d7da; padding: 12px; margin: 10px 0; border-left: 3px solid #d63638; border-radius: 3px;">';
            echo '<strong style="color: #721c24;">❌ Configuration Issues Detected</strong><br>';
            echo '<p style="margin: 5px 0; color: #721c24;">One or more CDN types are not configured correctly. Please review the issues above.</p>';
            echo '</div>';
            
            // Check specific issues for all CDN types
            $image_enabled = get_option('cdn1_enable_image_cdn', 0);
            $image_url = get_option('cdn1_cdn_image_base', '');
            $media_enabled = get_option('cdn1_enable_media_cdn', 0);
            $media_url = get_option('cdn1_cdn_media_base', '');
            $css_enabled = get_option('cdn1_enable_css_cdn', 0);
            $css_url = get_option('cdn1_cdn_css_base', '');
            $js_enabled = get_option('cdn1_enable_js_cdn', 0);
            $js_url = get_option('cdn1_cdn_js_base', '');
            $fonts_enabled = get_option('cdn1_enable_fonts_cdn', 0);
            $fonts_url = get_option('cdn1_cdn_fonts_base', '');
            
            $disabled_cdns = [];
            $unconfigured_cdns = [];
            
            if (!$image_enabled) $disabled_cdns[] = 'Image CDN';
            elseif (!$image_url) $unconfigured_cdns[] = 'Image CDN';
            
            if (!$media_enabled) $disabled_cdns[] = 'Media CDN';
            elseif (!$media_url) $unconfigured_cdns[] = 'Media CDN';
            
            if (!$css_enabled) $disabled_cdns[] = 'CSS CDN';
            elseif (!$css_url) $unconfigured_cdns[] = 'CSS CDN';
            
            if (!$js_enabled) $disabled_cdns[] = 'JavaScript CDN';
            elseif (!$js_url) $unconfigured_cdns[] = 'JavaScript CDN';
            
            if (!$fonts_enabled) $disabled_cdns[] = 'Fonts CDN';
            elseif (!$fonts_url) $unconfigured_cdns[] = 'Fonts CDN';
            
            if (!empty($disabled_cdns)) {
                echo '<div style="background: #fff3cd; padding: 10px; margin: 10px 0; border-left: 3px solid #ffc107;">';
                echo '<strong>⚠️ Disabled CDNs:</strong> ' . implode(', ', $disabled_cdns) . '<br>';
                echo '<p style="margin: 5px 0;">Please enable them in the <a href="' . esc_url(admin_url('admin.php?page=cdn1-settings')) . '">Settings</a> page.</p>';
                echo '</div>';
            }
            
            if (!empty($unconfigured_cdns)) {
                echo '<div style="background: #fff3cd; padding: 10px; margin: 10px 0; border-left: 3px solid #ffc107;">';
                echo '<strong>⚠️ CDN URLs Missing:</strong> ' . implode(', ', $unconfigured_cdns) . '<br>';
                echo '<p style="margin: 5px 0;">Please configure the CDN URLs in the <a href="' . esc_url(admin_url('admin.php?page=cdn1-settings')) . '">Settings</a> page.</p>';
                echo '</div>';
            }
            
            echo '<p style="margin: 10px 0;"><strong>Recommended Actions:</strong></p>';
            echo '<ol style="margin: 5px 0 5px 20px;">';
            echo '<li>Go to <a href="' . esc_url(admin_url('admin.php?page=cdn1-settings')) . '">Settings</a> and verify your CDN configuration</li>';
            echo '<li>Enable all required CDN types and provide their URLs</li>';
            echo '<li>Clear plugin cache using the "Clear All Cache" button above</li>';
            echo '<li>Run this diagnostic again to verify the fix</li>';
            echo '</ol>';
        }
        
        echo '</div>';
        echo '</div>';
        
        $html = ob_get_clean();
        
        wp_send_json_success([
            'html' => $html
        ]);
    }
    
    /**
     * AJAX handler to fix WordPress URLs
     */
    public function ajax_fix_wordpress_urls() {
        check_ajax_referer('cdn1_fix_wp_urls', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => 'Insufficient permissions']);
            return;
        }
        
        try {
            $current_home = get_option('home');
            $current_siteurl = get_option('siteurl');
            
            // Get the correct domain from SERVER_NAME
            $server_name = $_SERVER['SERVER_NAME'] ?? '';
            
            if (empty($server_name)) {
                wp_send_json_error(['message' => 'Cannot determine server name']);
                return;
            }
            
            // Build correct URLs using SERVER_NAME
            $protocol = is_ssl() ? 'https://' : 'http://';
            $correct_url = $protocol . $server_name;
            
            // Update WordPress core URLs
            update_option('home', $correct_url);
            update_option('siteurl', $correct_url);
            
            $message = sprintf(
                'WordPress URLs fixed! Changed from %s to %s',
                $current_home,
                $correct_url
            );
            
            wp_send_json_success([
                'message' => $message,
                'old_url' => $current_home,
                'new_url' => $correct_url
            ]);
        } catch (Exception $e) {
            wp_send_json_error([
                'message' => 'Error: ' . $e->getMessage()
            ]);
        }
    }
    
    /**
     * AJAX: Auto-inject meta tag
     */
    public function ajax_auto_inject_meta() {
        check_ajax_referer('cdn1_ajax', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => 'Permission denied']);
        }
        
        update_option('cdn1_auto_inject_meta', 1);
        
        wp_send_json_success([
            'message' => 'Meta tag auto-injection enabled!'
        ]);
    }
    
    /**
     * AJAX: Verify domain
     */
    public function ajax_verify_domain() {
        check_ajax_referer('cdn1_ajax', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => 'Permission denied']);
        }
        
        $method = isset($_POST['method']) ? sanitize_text_field($_POST['method']) : '';
        
        if (empty($method)) {
            wp_send_json_error(['message' => 'Invalid verification method']);
        }
        
        $result = CDN1_Security::verify_domain($method);
        
        if ($result['success']) {
            wp_send_json_success([
                'message' => $result['message'],
                'method' => $method,
                'details' => isset($result['details']) ? $result['details'] : ''
            ]);
        } else {
            wp_send_json_error([
                'message' => $result['message']
            ]);
        }
    }
    
    /**
     * AJAX: Complete registration
     */
    public function ajax_complete_registration() {
        check_ajax_referer('cdn1_ajax', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => 'Permission denied']);
        }
        
        $result = CDN1_Security::register_domain();
        
        if ($result['success']) {
            wp_send_json_success([
                'message' => $result['message'],
                'key' => isset($result['key']) ? $result['key'] : ''
            ]);
        } else {
            wp_send_json_error([
                'message' => $result['message']
            ]);
        }
    }
    
    /**
     * AJAX: Revoke registration
     */
    public function ajax_revoke_registration() {
        check_ajax_referer('cdn1_ajax', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => 'Permission denied']);
        }
        
        CDN1_Security::revoke_registration();
        
        wp_send_json_success([
            'message' => 'Registration revoked successfully. You can re-register anytime.'
        ]);
    }
    
    /**
     * AJAX handler to force update check
     */
    public function ajax_force_update_check() {
        check_ajax_referer('cdn1_force_update_check', 'nonce');
        
        if (!current_user_can('update_plugins')) {
            wp_send_json_error(['message' => 'You do not have permission to check for updates']);
            return;
        }
        
        // Force update check
        CDN1_Update_Checker::force_check();
        
        // Get update information from S3
        $update_url = 'https://awsmarketplace-public-personal.s3.us-east-1.amazonaws.com/1cdn-wordpress-plugin/update-info.json';
        $response = wp_remote_get($update_url, ['timeout' => 10]);
        
        if (is_wp_error($response)) {
            wp_send_json_error([
                'message' => 'Failed to connect to update server: ' . $response->get_error_message()
            ]);
            return;
        }
        
        $body = wp_remote_retrieve_body($response);
        $update_info = json_decode($body);
        
        if (json_last_error() !== JSON_ERROR_NONE) {
            wp_send_json_error([
                'message' => 'Failed to parse update information'
            ]);
            return;
        }
        
        $current_version = CDN1_VERSION;
        $latest_version = $update_info->version;
        
        $update_available = version_compare($current_version, $latest_version, '<');
        
        $data = [
            'current_version' => $current_version,
            'latest_version' => $latest_version,
            'update_available' => $update_available,
        ];
        
        if ($update_available) {
            $data['update_url'] = admin_url('plugins.php');
            $data['changelog'] = $update_info->sections->changelog ?? '';
        }
        
        wp_send_json_success($data);
    }
}
