<?php
/**
 * CDN CORS Handler
 * Adds proper CORS headers for fonts and static assets
 * Fixes 404 errors and cross-origin issues
 */

if (!defined('ABSPATH')) {
    exit;
}

class CDN1_CORS_Handler {
    
    private static $instance = null;
    
    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    private function __construct() {
        $this->init_hooks();
    }
    
    /**
     * Initialize hooks
     */
    private function init_hooks() {
        // Add CORS headers - use send_headers hook which runs after init
        add_action('send_headers', [$this, 'add_cors_headers'], 1);
        
        // Also add to template_redirect for frontend requests
        add_action('template_redirect', [$this, 'add_cors_headers'], 1);
        
        // Fix font MIME types
        add_filter('upload_mimes', [$this, 'add_font_mimes']);
        add_filter('wp_check_filetype_and_ext', [$this, 'fix_font_mime_type'], 10, 4);
        
        // Add cache directory .htaccess - runs after WordPress is fully loaded
        add_action('wp_loaded', [$this, 'ensure_cache_htaccess']);
        
        // Add rewrite rules after init
        add_action('init', [$this, 'add_rewrite_rules']);
        add_filter('query_vars', [$this, 'add_query_vars']);
        add_action('template_redirect', [$this, 'handle_font_request']);
    }
    
    /**
     * Add CORS headers for cross-origin requests
     */
    public function add_cors_headers() {
        // Ensure WordPress functions are available
        if (!function_exists('home_url') || !function_exists('get_option')) {
            return;
        }
        
        // Get all configured domains
        $configured_domains = $this->get_all_configured_domains();
        
        // Get current host
        $current_host = isset($_SERVER['HTTP_HOST']) ? sanitize_text_field(wp_unslash($_SERVER['HTTP_HOST'])) : '';
        $home_url = home_url();
        $main_host = $home_url ? wp_parse_url($home_url, PHP_URL_HOST) : '';
        
        // Check if we're on any configured subdomain/domain
        $is_alt_domain = false;
        foreach ($configured_domains as $domain) {
            $domain_host = wp_parse_url($domain, PHP_URL_HOST);
            if ($domain_host && strcasecmp($current_host, $domain_host) === 0) {
                $is_alt_domain = true;
                break;
            }
        }
        
        // Add CORS headers if on alternative domain OR accessing static assets
        if ($is_alt_domain || ($main_host && strcasecmp($current_host, $main_host) !== 0)) {
            $this->send_cors_headers();
        }
        
        // Always add CORS for font files and static assets
        $request_uri = isset($_SERVER['REQUEST_URI']) ? sanitize_text_field(wp_unslash($_SERVER['REQUEST_URI'])) : '';
        if ($this->is_static_asset($request_uri) || $this->is_font_file($request_uri)) {
            $this->send_cors_headers();
        }
    }
    
    /**
     * Get all configured domains from plugin settings
     */
    private function get_all_configured_domains() {
        $domains = [];
        
        // Origin URL
        if (get_option('cdn1_enable_origin_url')) {
            $origin_url = get_option('cdn1_origin_url', '');
            if ($origin_url) {
                $domains[] = $origin_url;
            }
        }
        
        // Admin URL
        if (get_option('cdn1_enable_admin_url')) {
            $admin_url = get_option('cdn1_admin_url', '');
            if ($admin_url) {
                $domains[] = $admin_url;
            }
        }
        
        // Image CDN
        if (get_option('cdn1_enable_image_cdn')) {
            $img_cdn = get_option('cdn1_cdn_image_base', '');
            if ($img_cdn) {
                $domains[] = $img_cdn;
            }
        }
        
        // Media CDN
        if (get_option('cdn1_enable_media_cdn')) {
            $media_cdn = get_option('cdn1_cdn_media_base', '');
            if ($media_cdn) {
                $domains[] = $media_cdn;
            }
        }
        
        // Static CDN
        if (get_option('cdn1_enable_static_cdn')) {
            $static_cdn = get_option('cdn1_cdn_static_base', '');
            if ($static_cdn) {
                $domains[] = $static_cdn;
            }
        }
        
        // Fonts CDN
        if (get_option('cdn1_enable_fonts_cdn')) {
            $fonts_cdn = get_option('cdn1_cdn_fonts_base', '');
            if ($fonts_cdn) {
                $domains[] = $fonts_cdn;
            }
        }
        
        // Frontend URL
        if (get_option('cdn1_enable_frontend_url')) {
            $frontend_url = get_option('cdn1_cdn_frontend_url', '');
            if ($frontend_url) {
                $domains[] = $frontend_url;
            }
        }
        
        return array_filter($domains);
    }
    
    /**
     * Send CORS headers
     */
    private function send_cors_headers() {
        if (headers_sent()) {
            return;
        }
        
        // Ensure WordPress functions are available
        if (!function_exists('home_url')) {
            // Fallback: just use wildcard if WordPress not ready
            header('Access-Control-Allow-Origin: *');
            header('Access-Control-Allow-Methods: GET, POST, OPTIONS');
            header('Access-Control-Allow-Headers: Content-Type, Authorization, X-Requested-With, X-Cache-Flush');
            header('Access-Control-Allow-Credentials: false');
            header('Access-Control-Max-Age: 86400');
            
            if (isset($_SERVER['REQUEST_METHOD']) && $_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
                status_header(200);
                exit;
            }
            return;
        }
        
        // Get origin from request
        $origin = isset($_SERVER['HTTP_ORIGIN']) ? sanitize_text_field(wp_unslash($_SERVER['HTTP_ORIGIN'])) : '';
        
        // Get all configured domains
        $configured_domains = $this->get_all_configured_domains();
        $home_url = home_url();
        $allowed_origins = $home_url ? array_merge($configured_domains, [$home_url]) : $configured_domains;
        
        // Check if origin is allowed
        $origin_allowed = false;
        foreach ($allowed_origins as $allowed_url) {
            if (!$allowed_url) continue;
            
            $allowed_host = wp_parse_url($allowed_url, PHP_URL_HOST);
            $origin_host = wp_parse_url($origin, PHP_URL_HOST);
            
            if ($allowed_host && $origin_host && strcasecmp($allowed_host, $origin_host) === 0) {
                $origin_allowed = true;
                break;
            }
        }
        
        // If origin is in our list, use it; otherwise use wildcard for CDN compatibility
        if ($origin_allowed && $origin) {
            header('Access-Control-Allow-Origin: ' . $origin);
            header('Vary: Origin');
        } else {
            // Fallback to wildcard for maximum CDN compatibility
            header('Access-Control-Allow-Origin: *');
        }
        
        header('Access-Control-Allow-Methods: GET, POST, OPTIONS');
        header('Access-Control-Allow-Headers: Content-Type, Authorization, X-Requested-With, X-Cache-Flush');
        header('Access-Control-Allow-Credentials: false');
        header('Access-Control-Max-Age: 86400'); // 24 hours
        
        // Handle preflight OPTIONS request
        if (isset($_SERVER['REQUEST_METHOD']) && $_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
            status_header(200);
            exit;
        }
    }
    
    /**
     * Check if request is for a static asset
     */
    private function is_static_asset($uri) {
        $static_extensions = [
            // Images
            'css', 'js', 'jpg', 'jpeg', 'png', 'gif', 'webp', 'svg', 'ico', 'bmp', 'tiff', 'avif',
            // Fonts
            'woff', 'woff2', 'ttf', 'otf', 'eot',
            // Media
            'mp4', 'webm', 'ogg', 'ogv', 'avi', 'mov', 'wmv', 'flv', 'mkv', 'm4v',
            'mp3', 'wav', 'm4a', 'aac', 'flac', 'oga',
            // Documents
            'pdf', 'doc', 'docx', 'xls', 'xlsx', 'ppt', 'pptx',
            // Archives
            'zip', 'rar', '7z', 'tar', 'gz'
        ];
        
        $ext = strtolower(pathinfo($uri, PATHINFO_EXTENSION));
        return in_array($ext, $static_extensions);
    }
    
    /**
     * Check if request is for a font file
     */
    private function is_font_file($uri) {
        $font_extensions = ['woff', 'woff2', 'ttf', 'otf', 'eot', 'svg'];
        $ext = strtolower(pathinfo($uri, PATHINFO_EXTENSION));
        return in_array($ext, $font_extensions);
    }
    
    /**
     * Add font MIME types
     */
    public function add_font_mimes($mimes) {
        $mimes['woff'] = 'font/woff';
        $mimes['woff2'] = 'font/woff2';
        $mimes['ttf'] = 'font/ttf';
        $mimes['otf'] = 'font/otf';
        $mimes['eot'] = 'application/vnd.ms-fontobject';
        $mimes['svg'] = 'image/svg+xml';
        
        return $mimes;
    }
    
    /**
     * Fix font MIME type detection
     */
    public function fix_font_mime_type($data, $file, $filename, $mimes) {
        $ext = strtolower(pathinfo($filename, PATHINFO_EXTENSION));
        
        $font_mimes = [
            'woff' => 'font/woff',
            'woff2' => 'font/woff2',
            'ttf' => 'font/ttf',
            'otf' => 'font/otf',
            'eot' => 'application/vnd.ms-fontobject',
        ];
        
        if (isset($font_mimes[$ext])) {
            $data['ext'] = $ext;
            $data['type'] = $font_mimes[$ext];
            $data['proper_filename'] = $filename;
        }
        
        return $data;
    }
    
    /**
     * Add rewrite rules for fonts and static assets
     */
    public function add_rewrite_rules() {
        // No special rewrite rules needed - fonts served from their original locations
        // CORS headers are added via send_headers hook
    }
    
    /**
     * Add query vars
     */
    public function add_query_vars($vars) {
        // No custom query vars needed
        return $vars;
    }
    
    /**
     * Handle font file requests
     * Note: This is intentionally a no-op as fonts are served from original locations
     */
    public function handle_font_request() {
        // Fonts are served directly from their original locations (theme/plugin directories)
        // with CORS headers added via send_headers hook
        return;
    }
    
    /**
     * Ensure cache directories have proper .htaccess
     */
    public function ensure_cache_htaccess() {
        // Ensure WordPress constants are defined
        if (!defined('WP_CONTENT_DIR')) {
            return;
        }
        
        // Add .htaccess to wp-content root for CORS on all fonts
        $this->ensure_wp_content_htaccess();
        
        // Add .htaccess for CSS cache directory
        $cache_dirs = [
            WP_CONTENT_DIR . '/cache/uscdn-css'
        ];
        
        foreach ($cache_dirs as $dir) {
            try {
                if (!is_dir($dir)) {
                    wp_mkdir_p($dir);
                }
                
                $htaccess_file = $dir . '/.htaccess';
                
                if (!file_exists($htaccess_file)) {
                    $htaccess_content = $this->get_htaccess_content();
                    @file_put_contents($htaccess_file, $htaccess_content);
                }
            } catch (Exception $e) {
                // Silently fail - not critical
                continue;
            }
        }
    }
    
    /**
     * Ensure wp-content directory has CORS headers for fonts
     */
    private function ensure_wp_content_htaccess() {
        $htaccess_file = WP_CONTENT_DIR . '/.htaccess';
        
        // Check if .htaccess exists
        $existing_content = '';
        if (file_exists($htaccess_file)) {
            $existing_content = @file_get_contents($htaccess_file);
            
            // Remove old CDN rules if they exist
            if (strpos($existing_content, '# Utility Server CDN - CORS Headers') !== false) {
                // Remove everything from our marker onwards
                $parts = explode('# Utility Server CDN - CORS Headers', $existing_content);
                $existing_content = $parts[0]; // Keep everything before our rules
            }
        }
        
        // Add complete CORS headers for all file types
        $cors_rules = "# Utility Server CDN - CORS Headers\n";
        $cors_rules .= "<IfModule mod_headers.c>\n";
        $cors_rules .= "    # Add CORS headers for font files\n";
        $cors_rules .= "    <FilesMatch \"\.(woff|woff2|ttf|otf|eot|svg)$\">\n";
        $cors_rules .= "        Header always set Access-Control-Allow-Origin \"*\"\n";
        $cors_rules .= "        Header always set Access-Control-Allow-Methods \"GET, OPTIONS\"\n";
        $cors_rules .= "        Header always set Access-Control-Allow-Headers \"Content-Type\"\n";
        $cors_rules .= "    </FilesMatch>\n";
        $cors_rules .= "    \n";
        $cors_rules .= "    # Add CORS headers for CSS and JS files\n";
        $cors_rules .= "    <FilesMatch \"\.(css|js)$\">\n";
        $cors_rules .= "        Header always set Access-Control-Allow-Origin \"*\"\n";
        $cors_rules .= "        Header always set Access-Control-Allow-Methods \"GET, OPTIONS\"\n";
        $cors_rules .= "        Header always set Access-Control-Allow-Headers \"Content-Type\"\n";
        $cors_rules .= "    </FilesMatch>\n";
        $cors_rules .= "    \n";
        $cors_rules .= "    # Add CORS headers for media files (video/audio)\n";
        $cors_rules .= "    <FilesMatch \"\.(mp4|webm|ogg|ogv|avi|mov|wmv|mp3|wav|m4a|aac|flac)$\">\n";
        $cors_rules .= "        Header always set Access-Control-Allow-Origin \"*\"\n";
        $cors_rules .= "        Header always set Access-Control-Allow-Methods \"GET, OPTIONS\"\n";
        $cors_rules .= "        Header always set Access-Control-Allow-Headers \"Content-Type, Range\"\n";
        $cors_rules .= "        Header always set Access-Control-Expose-Headers \"Accept-Ranges, Content-Encoding, Content-Length, Content-Range\"\n";
        $cors_rules .= "    </FilesMatch>\n";
        $cors_rules .= "</IfModule>\n";
        $cors_rules .= "\n";
        $cors_rules .= "# Font and Media MIME types\n";
        $cors_rules .= "<IfModule mod_mime.c>\n";
        $cors_rules .= "    AddType font/woff .woff\n";
        $cors_rules .= "    AddType font/woff2 .woff2\n";
        $cors_rules .= "    AddType font/ttf .ttf\n";
        $cors_rules .= "    AddType font/otf .otf\n";
        $cors_rules .= "    AddType application/vnd.ms-fontobject .eot\n";
        $cors_rules .= "    AddType video/mp4 .mp4\n";
        $cors_rules .= "    AddType video/webm .webm\n";
        $cors_rules .= "    AddType audio/mpeg .mp3\n";
        $cors_rules .= "    AddType audio/wav .wav\n";
        $cors_rules .= "</IfModule>\n";
        
        // Append new rules to existing content
        $new_content = rtrim($existing_content) . "\n\n" . ltrim($cors_rules);
        
        // Write the file
        @file_put_contents($htaccess_file, $new_content);
    }
    
    /**
     * Get .htaccess content for cache directories
     */
    private function get_htaccess_content() {
        $htaccess = "# 1 CDN Manager - Cache Directory\n";
        $htaccess .= "# Enable direct file access and CORS\n\n";
        $htaccess .= "<IfModule mod_headers.c>\n";
        $htaccess .= "    # CORS headers for all files\n";
        $htaccess .= "    Header set Access-Control-Allow-Origin \"*\"\n";
        $htaccess .= "    Header set Access-Control-Allow-Methods \"GET, OPTIONS\"\n";
        $htaccess .= "    Header set Access-Control-Allow-Headers \"Content-Type\"\n";
        $htaccess .= "    \n";
        $htaccess .= "    # Cache headers\n";
        $htaccess .= "    Header set Cache-Control \"public, max-age=31536000\"\n";
        $htaccess .= "</IfModule>\n\n";
        $htaccess .= "# Font MIME types\n";
        $htaccess .= "<IfModule mod_mime.c>\n";
        $htaccess .= "    AddType font/woff .woff\n";
        $htaccess .= "    AddType font/woff2 .woff2\n";
        $htaccess .= "    AddType font/ttf .ttf\n";
        $htaccess .= "    AddType font/otf .otf\n";
        $htaccess .= "    AddType application/vnd.ms-fontobject .eot\n";
        $htaccess .= "    AddType image/svg+xml .svg\n";
        $htaccess .= "</IfModule>\n\n";
        $htaccess .= "# Enable file access\n";
        $htaccess .= "<IfModule mod_rewrite.c>\n";
        $htaccess .= "    RewriteEngine On\n";
        $htaccess .= "    RewriteBase /\n";
        $htaccess .= "</IfModule>\n\n";
        $htaccess .= "# Allow from all\n";
        $htaccess .= "<IfModule mod_authz_core.c>\n";
        $htaccess .= "    Require all granted\n";
        $htaccess .= "</IfModule>\n";
        $htaccess .= "<IfModule !mod_authz_core.c>\n";
        $htaccess .= "    Order allow,deny\n";
        $htaccess .= "    Allow from all\n";
        $htaccess .= "</IfModule>\n\n";
        $htaccess .= "# Enable compression\n";
        $htaccess .= "<IfModule mod_deflate.c>\n";
        $htaccess .= "    AddOutputFilterByType DEFLATE text/css\n";
        $htaccess .= "    AddOutputFilterByType DEFLATE font/woff\n";
        $htaccess .= "    AddOutputFilterByType DEFLATE font/woff2\n";
        $htaccess .= "</IfModule>\n";
        
        return $htaccess;
    }
}
