<?php
/**
 * CDN Content Migrator
 * Handles search/replace of URLs in saved content (posts, pages, custom fields)
 */

if (!defined('ABSPATH')) {
    exit;
}

class CDN1_Migrator {
    
    private static $instance = null;
    
    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    private function __construct() {
        // AJAX handlers
        add_action('wp_ajax_cdn1_migrate_content', [$this, 'ajax_migrate_content']);
        add_action('wp_ajax_cdn1_scan_content', [$this, 'ajax_scan_content']);
    }
    
    /**
     * Scan database for URLs that need migration
     */
    public static function scan_urls($old_url, $new_url) {
        global $wpdb;
        
        $old_url = trim($old_url);
        $new_url = trim($new_url);
        
        if (empty($old_url) || empty($new_url)) {
            return ['error' => 'Both URLs are required'];
        }
        
        $results = [
            'posts' => 0,
            'postmeta' => 0,
            'options' => 0,
            'comments' => 0,
            'total' => 0
        ];
        
        // Scan posts (post_content, post_excerpt, guid)
        $posts = $wpdb->get_var($wpdb->prepare(
            "SELECT COUNT(*) FROM {$wpdb->posts} 
            WHERE post_content LIKE %s 
            OR post_excerpt LIKE %s 
            OR guid LIKE %s",
            '%' . $wpdb->esc_like($old_url) . '%',
            '%' . $wpdb->esc_like($old_url) . '%',
            '%' . $wpdb->esc_like($old_url) . '%'
        ));
        $results['posts'] = (int) $posts;
        
        // Scan postmeta
        $postmeta = $wpdb->get_var($wpdb->prepare(
            "SELECT COUNT(*) FROM {$wpdb->postmeta} 
            WHERE meta_value LIKE %s",
            '%' . $wpdb->esc_like($old_url) . '%'
        ));
        $results['postmeta'] = (int) $postmeta;
        
        // Scan options
        $options = $wpdb->get_var($wpdb->prepare(
            "SELECT COUNT(*) FROM {$wpdb->options} 
            WHERE option_value LIKE %s
            AND option_name NOT LIKE %s",
            '%' . $wpdb->esc_like($old_url) . '%',
            '%transient%'
        ));
        $results['options'] = (int) $options;
        
        // Scan comments
        $comments = $wpdb->get_var($wpdb->prepare(
            "SELECT COUNT(*) FROM {$wpdb->comments} 
            WHERE comment_content LIKE %s 
            OR comment_author_url LIKE %s",
            '%' . $wpdb->esc_like($old_url) . '%',
            '%' . $wpdb->esc_like($old_url) . '%'
        ));
        $results['comments'] = (int) $comments;
        
        $results['total'] = $results['posts'] + $results['postmeta'] + $results['options'] + $results['comments'];
        
        return $results;
    }
    
    /**
     * Migrate URLs in database
     */
    public static function migrate_urls($old_url, $new_url, $dry_run = false) {
        global $wpdb;
        
        $old_url = trim($old_url);
        $new_url = trim($new_url);
        
        if (empty($old_url) || empty($new_url)) {
            return ['error' => 'Both URLs are required'];
        }
        
        $results = [
            'posts' => 0,
            'postmeta' => 0,
            'options' => 0,
            'comments' => 0,
            'total' => 0,
            'dry_run' => $dry_run
        ];
        
        if ($dry_run) {
            return self::scan_urls($old_url, $new_url);
        }
        
        // Update posts (post_content, post_excerpt)
        $posts = $wpdb->query($wpdb->prepare(
            "UPDATE {$wpdb->posts} 
            SET post_content = REPLACE(post_content, %s, %s),
                post_excerpt = REPLACE(post_excerpt, %s, %s)
            WHERE post_content LIKE %s 
            OR post_excerpt LIKE %s",
            $old_url, $new_url,
            $old_url, $new_url,
            '%' . $wpdb->esc_like($old_url) . '%',
            '%' . $wpdb->esc_like($old_url) . '%'
        ));
        $results['posts'] = (int) $posts;
        
        // Update GUIDs (optional, usually not recommended)
        // Skip GUIDs for now as WordPress recommends not changing them
        
        // Update postmeta
        $postmeta = $wpdb->query($wpdb->prepare(
            "UPDATE {$wpdb->postmeta} 
            SET meta_value = REPLACE(meta_value, %s, %s)
            WHERE meta_value LIKE %s",
            $old_url, $new_url,
            '%' . $wpdb->esc_like($old_url) . '%'
        ));
        $results['postmeta'] = (int) $postmeta;
        
        // Update options (skip transients)
        $options = $wpdb->query($wpdb->prepare(
            "UPDATE {$wpdb->options} 
            SET option_value = REPLACE(option_value, %s, %s)
            WHERE option_value LIKE %s
            AND option_name NOT LIKE %s",
            $old_url, $new_url,
            '%' . $wpdb->esc_like($old_url) . '%',
            '%transient%'
        ));
        $results['options'] = (int) $options;
        
        // Update comments
        $comments = $wpdb->query($wpdb->prepare(
            "UPDATE {$wpdb->comments} 
            SET comment_content = REPLACE(comment_content, %s, %s),
                comment_author_url = REPLACE(comment_author_url, %s, %s)
            WHERE comment_content LIKE %s 
            OR comment_author_url LIKE %s",
            $old_url, $new_url,
            $old_url, $new_url,
            '%' . $wpdb->esc_like($old_url) . '%',
            '%' . $wpdb->esc_like($old_url) . '%'
        ));
        $results['comments'] = (int) $comments;
        
        $results['total'] = $results['posts'] + $results['postmeta'] + $results['options'] + $results['comments'];
        
        // Clear cache after migration
        wp_cache_flush();
        
        return $results;
    }
    
    /**
     * AJAX handler for content migration
     */
    public function ajax_migrate_content() {
        check_ajax_referer('cdn1_admin', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => 'Unauthorized']);
        }
        
        $old_url = isset($_POST['old_url']) ? esc_url_raw($_POST['old_url']) : '';
        $new_url = isset($_POST['new_url']) ? esc_url_raw($_POST['new_url']) : '';
        $dry_run = isset($_POST['dry_run']) && $_POST['dry_run'] === 'true';
        
        if (empty($old_url) || empty($new_url)) {
            wp_send_json_error(['message' => 'Both URLs are required']);
        }
        
        $results = self::migrate_urls($old_url, $new_url, $dry_run);
        
        if (isset($results['error'])) {
            wp_send_json_error($results);
        } else {
            wp_send_json_success($results);
        }
    }
    
    /**
     * AJAX handler for content scanning
     */
    public function ajax_scan_content() {
        check_ajax_referer('cdn1_admin', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => 'Unauthorized']);
        }
        
        $old_url = isset($_POST['old_url']) ? esc_url_raw($_POST['old_url']) : '';
        $new_url = isset($_POST['new_url']) ? esc_url_raw($_POST['new_url']) : '';
        
        if (empty($old_url) || empty($new_url)) {
            wp_send_json_error(['message' => 'Both URLs are required']);
        }
        
        $results = self::scan_urls($old_url, $new_url);
        
        if (isset($results['error'])) {
            wp_send_json_error($results);
        } else {
            wp_send_json_success($results);
        }
    }
    
    /**
     * Get common URL migrations based on current settings
     */
    public static function get_suggested_migrations() {
        $suggestions = [];
        
        $current_url = home_url();
        $origin_url = CDN1_Settings::get('origin_url', '');
        $image_cdn = CDN1_Settings::get('cdn_image_base', '');
        $media_cdn = CDN1_Settings::get('cdn_media_base', '');
        $static_cdn = CDN1_Settings::get('cdn_static_base', '');
        $fonts_cdn = CDN1_Settings::get('cdn_fonts_base', '');
        
        // Suggest origin URL migration
        if (!empty($origin_url) && $origin_url !== $current_url) {
            $suggestions[] = [
                'from' => $current_url,
                'to' => $origin_url,
                'type' => 'origin',
                'description' => 'Migrate content URLs to origin server'
            ];
        }
        
        // Suggest image CDN migration
        if (!empty($image_cdn)) {
            $suggestions[] = [
                'from' => $current_url . '/wp-content/uploads',
                'to' => $image_cdn . '/wp-content/uploads',
                'type' => 'image',
                'description' => 'Migrate image URLs to Image CDN'
            ];
        }
        
        // Suggest media CDN migration
        if (!empty($media_cdn)) {
            $suggestions[] = [
                'from' => $current_url . '/wp-content/uploads',
                'to' => $media_cdn . '/wp-content/uploads',
                'type' => 'media',
                'description' => 'Migrate media (video/audio) URLs to Media CDN'
            ];
        }
        
        // Suggest static CDN migration
        if (!empty($static_cdn)) {
            $suggestions[] = [
                'from' => $current_url . '/wp-content',
                'to' => $static_cdn . '/wp-content',
                'type' => 'static',
                'description' => 'Migrate static resource (JS/CSS) URLs to Static CDN'
            ];
        }
        
        // Suggest fonts CDN migration
        if (!empty($fonts_cdn)) {
            $suggestions[] = [
                'from' => $current_url . '/wp-content',
                'to' => $fonts_cdn . '/wp-content',
                'type' => 'fonts',
                'description' => 'Migrate fonts and binary file URLs to Fonts CDN'
            ];
        }
        
        return $suggestions;
    }
}
