<?php
/**
 * CDN URL Replacer
 * Handles all URL replacements for CDN integration
 */

if (!defined('ABSPATH')) {
    exit;
}

class CDN1_Replacer {
    
    private static $instance = null;
    private static $html_processed = false; // Flag to prevent multiple processing
    
    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    private function __construct() {
        $this->init_filters();
    }
    
    private function init_filters() {
        // Early URL builder filters
        add_filter('wp_get_attachment_url', [$this, 'replace_url'], 10, 1);
        add_filter('style_loader_src', [$this, 'replace_url'], 10, 1);
        add_filter('script_loader_src', [$this, 'replace_url'], 10, 1);
        add_filter('theme_file_uri', [$this, 'replace_url'], 10, 1);
        add_filter('template_directory_uri', [$this, 'replace_url'], 10, 1);
        add_filter('stylesheet_uri', [$this, 'replace_url'], 10, 1);
        add_filter('stylesheet_directory_uri', [$this, 'replace_url'], 10, 1);
        add_filter('content_url', [$this, 'replace_url'], 10, 1);
        add_filter('includes_url', [$this, 'replace_url'], 10, 1);
        add_filter('plugins_url', [$this, 'replace_url'], 10, 1);
        
        // Image specific filters
        add_filter('wp_calculate_image_srcset', [$this, 'replace_srcset']);
        add_filter('wp_get_attachment_image_src', [$this, 'replace_image_src'], 10, 4);
        
        // Loader tag safety nets
        add_filter('style_loader_tag', [$this, 'fix_style_tag'], 10, 4);
        add_filter('script_loader_tag', [$this, 'fix_script_tag'], 10, 3);
        
        // Content filters
        add_filter('the_content', [$this, 'replace_in_html'], 11);
        add_filter('widget_text', [$this, 'replace_in_html'], 11);
        add_filter('pagelayer_content', [$this, 'replace_in_html'], 11);
        
        // Output buffer for final HTML
        add_action('template_redirect', [$this, 'start_output_buffer']);
    }
    
    /**
     * Handle Default CDN URL replacement
     * Format: https://[subdomain].1cdn.us/https://original-domain.com/path
     */
    private function handle_default_cdn_url($url) {
        // Check if URL is already a Default CDN URL (prevent double wrapping)
        if (preg_match('/^https?:\/\/(js|cs|im|ft|mi)\.1cdn\.us\//i', $url)) {
            return $url; // Already processed, skip
        }
        
        // Normalize protocol-relative & root-relative URLs
        $parsed = @wp_parse_url($url);
        $path = $parsed['path'] ?? '';
        
        $origin = $this->get_origin();
        
        if (empty($parsed['host'])) {
            if (str_starts_with($url, '//')) {
                $url = $origin['scheme'] . ltrim($url, '/');
            }
            if (str_starts_with($url, '/')) {
                $url = $origin['full'] . $url;
            }
            $parsed = @wp_parse_url($url);
            $path = $parsed['path'] ?? $path;
        }
        
        $url_host = $parsed['host'] ?? '';
        
        // Check if URL is from a Default CDN subdomain - if so, skip
        if (preg_match('/^(js|cs|im|ft|mi)\.1cdn\.us$/i', $url_host)) {
            return $url; // Already a Default CDN URL
        }
        
        // Only rewrite if the URL comes from our known site/origin hosts
        if (!$this->is_from_source_host($url_host)) {
            return $url;
        }
        
        // Admin-like contexts: skip LINK rewrites, but DO rewrite static assets
        $is_admin_like = $this->is_admin_context();
        if ($is_admin_like && !$this->is_static_asset_path($path)) {
            return $url;
        }
        
        // Determine asset type and check if enabled
        $asset_type = null;
        $cdn_subdomain = null;
        
        if ($this->is_js_file($path) && CDN1_Settings::is_default_cdn_type_enabled('js')) {
            $asset_type = 'js';
            $cdn_subdomain = CDN1_Settings::get_default_cdn_subdomain('js');
        } elseif ($this->is_css_file($path) && CDN1_Settings::is_default_cdn_type_enabled('css')) {
            $asset_type = 'css';
            $cdn_subdomain = CDN1_Settings::get_default_cdn_subdomain('css');
        } elseif ($this->is_image_file($path) && CDN1_Settings::is_default_cdn_type_enabled('image')) {
            $asset_type = 'image';
            $cdn_subdomain = CDN1_Settings::get_default_cdn_subdomain('image');
        } elseif ($this->is_fonts_file($path) && CDN1_Settings::is_default_cdn_type_enabled('fonts')) {
            $asset_type = 'fonts';
            $cdn_subdomain = CDN1_Settings::get_default_cdn_subdomain('fonts');
        } elseif ($this->is_media_file($path) && CDN1_Settings::is_default_cdn_type_enabled('media')) {
            $asset_type = 'media';
            $cdn_subdomain = CDN1_Settings::get_default_cdn_subdomain('media');
        }
        
        // If no matching asset type enabled, return original URL
        if (!$asset_type || !$cdn_subdomain) {
            return $url;
        }
        
        // Build Default CDN URL: https://[subdomain].1cdn.us/https://original-url
        $scheme = is_ssl() ? 'https://' : 'http://';
        
        // Ensure the original URL is absolute
        if (!str_starts_with($url, 'http://') && !str_starts_with($url, 'https://')) {
            $url = $origin['full'] . '/' . ltrim($url, '/');
        }
        
        $default_cdn_url = $scheme . $cdn_subdomain . '/' . $url;
        
        // Add versioning
        $default_cdn_url = $this->add_versioning($default_cdn_url, $asset_type);
        
        return $default_cdn_url;
    }
    
    /**
     * Main URL replacement function - CORE CDN LOGIC
     * Based on custom_cdn_url_replacer from original functions.php
     */
    public function replace_url($url) {
        if (!$url || !is_string($url)) {
            return $url;
        }
        
        // CRITICAL: Never CDN admin JavaScript files - they need fresh versions
        if (strpos($url, '/wp-content/plugins/') !== false && strpos($url, '/admin/js/') !== false) {
            return $url; // Return original URL without CDN
        }
        
        // CRITICAL: Never CDN wp-admin files
        if (strpos($url, '/wp-admin/') !== false) {
            return $url;
        }
        
        // Check for Default CDN first - it takes priority over everything
        if (CDN1_Settings::is_default_cdn_enabled()) {
            return $this->handle_default_cdn_url($url);
        }
        
        $origin = $this->get_origin();
        $bases = CDN1_Settings::get_cdn_bases();
        
        // Normalize protocol-relative & root-relative URLs
        $parsed = @wp_parse_url($url);
        $path = $parsed['path'] ?? '';
        
        if (empty($parsed['host'])) {
            if (str_starts_with($url, '//')) {
                $url = $origin['scheme'] . ltrim($url, '/');
            }
            if (str_starts_with($url, '/')) {
                $url = $origin['full'] . $url;
            }
            $parsed = @wp_parse_url($url);
            $path = $parsed['path'] ?? $path;
        }
        
        $url_host = $parsed['host'] ?? '';
        // Only rewrite if the URL comes from our known site/origin hosts
        if (!$this->is_from_source_host($url_host)) {
            return $url;
        }
        
        // Admin-like contexts: skip LINK rewrites, but DO rewrite static assets
        $is_admin_like = $this->is_admin_context();
        if ($is_admin_like && !$this->is_static_asset_path($path)) {
            return $url;
        }
        
        // 1) Images → Image CDN
        if (CDN1_Settings::get('enable_image_cdn') && $this->is_image_file($path)) {
            return $this->replace_base($url, $bases['img'], 'image');
        }
        
        // 2) Media (Video/Audio) → Media CDN
        if (CDN1_Settings::get('enable_media_cdn') && $this->is_media_file($path)) {
            return $this->replace_base($url, $bases['media'], 'media');
        }
        
        // 3) CSS files → CSS CDN
        if (CDN1_Settings::get('enable_css_cdn') && $this->is_css_file($path)) {
            return $this->replace_base($url, $bases['css'], 'css');
        }
        
        // 4) JavaScript files → JS CDN
        if (CDN1_Settings::get('enable_js_cdn') && $this->is_js_file($path)) {
            return $this->replace_base($url, $bases['js'], 'js');
        }
        
        // Legacy: Static assets (JS/CSS) → Static CDN (for backward compatibility)
        if (CDN1_Settings::get('enable_static_cdn') && $this->is_static_asset_path($path)) {
            // Determine if it's CSS or JS
            $cdn_type = $this->is_css_file($path) ? 'css' : 'js';
            return $this->replace_base($url, $bases['static'], $cdn_type);
        }
        
        // 5) Fonts and Binary files → Fonts CDN
        if (CDN1_Settings::get('enable_fonts_cdn') && $this->is_fonts_file($path)) {
            return $this->replace_base($url, $bases['fonts'], 'fonts');
        }
        
        // 6) Frontend URL (frontend only — admin-like non-static already returned above)
        if (CDN1_Settings::get('enable_frontend_url')) {
            // Detect file type for proper versioning even through frontend URL
            $cdn_type = 'frontend';
            if ($this->is_image_file($path)) {
                $cdn_type = 'image';
            } elseif ($this->is_media_file($path)) {
                $cdn_type = 'media';
            } elseif ($this->is_css_file($path)) {
                $cdn_type = 'css';
            } elseif ($this->is_js_file($path)) {
                $cdn_type = 'js';
            } elseif ($this->is_fonts_file($path)) {
                $cdn_type = 'fonts';
            }
            return $this->replace_base($url, $bases['frontend'], $cdn_type);
        }
        
        // IMPORTANT: Apply versioning even if no CDN is enabled (cache busting)
        // This ensures browser cache is busted when version is updated
        $cdn_type = null;
        if ($this->is_image_file($path)) {
            $cdn_type = 'image';
        } elseif ($this->is_media_file($path)) {
            $cdn_type = 'media';
        } elseif ($this->is_css_file($path)) {
            $cdn_type = 'css';
        } elseif ($this->is_js_file($path)) {
            $cdn_type = 'js';
        } elseif ($this->is_fonts_file($path)) {
            $cdn_type = 'fonts';
        }
        
        if ($cdn_type) {
            return $this->add_versioning($url, $cdn_type);
        }
        
        return $url;
    }
    
    /**
     * Get origin information
     */
    private function get_origin() {
        $host = isset($_SERVER['HTTP_HOST']) ? sanitize_text_field(wp_unslash($_SERVER['HTTP_HOST'])) : '';
        $scheme = is_ssl() ? 'https://' : 'http://';
        return [
            'full' => $scheme . $host,
            'host' => $host,
            'scheme' => $scheme
        ];
    }
    
    /**
     * Check if URL is from source host
     */
    private function is_from_source_host($url_host) {
        if (empty($url_host)) {
            return true;
        }
        
        $sources = CDN1_Settings::get_source_hosts();
        foreach ($sources as $h) {
            if (strcasecmp($h, $url_host) === 0) {
                return true;
            }
        }
        return false;
    }
    
    /**
     * Check if in admin context
     */
    private function is_admin_context() {
        return (function_exists('is_admin') && is_admin()) 
            || (defined('DOING_AJAX') && DOING_AJAX)
            || (isset($_SERVER['REQUEST_URI']) && str_contains(sanitize_text_field(wp_unslash($_SERVER['REQUEST_URI'])), 'wp-login.php'));
    }
    
    
    /**
     * Check if path is an image file
     */
    private function is_image_file($path) {
        if (!$path) return false;
        
        $image_extensions = [
            '.jpg', '.jpeg', '.png', '.gif', '.webp', '.svg', 
            '.ico', '.bmp', '.tiff', '.tif', '.avif'
        ];
        
        $path_lower = strtolower($path);
        foreach ($image_extensions as $ext) {
            if (str_ends_with($path_lower, $ext)) {
                return true;
            }
        }
        
        return false;
    }
    
    /**
     * Check if path is a media file (video/audio)
     */
    private function is_media_file($path) {
        if (!$path) return false;
        
        $media_extensions = [
            '.mp4', '.webm', '.ogg', '.ogv', '.avi', '.mov', 
            '.wmv', '.flv', '.mkv', '.m4v',
            '.mp3', '.wav', '.m4a', '.aac', '.flac', '.oga'
        ];
        
        $path_lower = strtolower($path);
        foreach ($media_extensions as $ext) {
            if (str_ends_with($path_lower, $ext)) {
                return true;
            }
        }
        
        return false;
    }

    /**
     * Check if path is a static asset (JS/CSS)
     */
    private function is_static_asset_path($path) {
        return $this->is_js_file($path) || $this->is_css_file($path);
    }
    
    /**
     * Check if path is a CSS file
     */
    private function is_css_file($path) {
        return str_ends_with(strtolower($path), '.css');
    }
    
    /**
     * Check if path is a JS file
     */
    private function is_js_file($path) {
        return str_ends_with(strtolower($path), '.js');
    }
    
    /**
     * Check if path is a fonts or binary file
     */
    private function is_fonts_file($path) {
        if (!$path) return false;
        
        // Fonts and binary files (PDFs, ZIPs, etc.)
        $fonts_extensions = [
            '.woff2', '.woff', '.ttf', '.otf', '.eot',
            '.pdf', '.doc', '.docx', '.xls', '.xlsx', 
            '.ppt', '.pptx', '.zip', '.rar', '.7z', 
            '.tar', '.gz', '.txt', '.csv'
        ];
        $path_lower = strtolower($path);
        
        foreach ($fonts_extensions as $ext) {
            if (str_ends_with($path_lower, $ext)) {
                return true;
            }
        }
        
        return false;
    }
    
    /**
     * Replace URL base
     */
    private function replace_base($url, $new_base, $cdn_type = null) {
        if (empty($new_base)) {
            return $url;
        }
        
        $p = @wp_parse_url($url);
        if (!$p || empty($p['path'])) {
            return $url;
        }
        
        $base = rtrim($new_base, '/');
        $path = '/' . ltrim($p['path'], '/');
        $query = isset($p['query']) ? ('?' . $p['query']) : '';
        $fragment = isset($p['fragment']) ? ('#' . $p['fragment']) : '';
        
        $new_url = $base . $path . $query . $fragment;
        
        // Add versioning if enabled (with CDN type)
        $new_url = $this->add_versioning($new_url, $cdn_type);
        
        return $new_url;
    }
    
    /**
     * Add versioning to URL for cache busting
     * 
     * @param string $url The URL to add version to
     * @param string $cdn_type The CDN type: 'image', 'media', 'fonts', 'css', 'js', 'frontend'
     */
    private function add_versioning($url, $cdn_type = null) {
        // Check if versioning is enabled for this CDN type
        $version = null;
        
        switch ($cdn_type) {
            case 'image':
                if (CDN1_Settings::get('enable_image_versioning')) {
                    $version = CDN1_Settings::get('image_version_string');
                }
                break;
            case 'media':
                if (CDN1_Settings::get('enable_media_versioning')) {
                    $version = CDN1_Settings::get('media_version_string');
                }
                break;
            case 'fonts':
                if (CDN1_Settings::get('enable_fonts_versioning')) {
                    $version = CDN1_Settings::get('fonts_version_string');
                }
                break;
            case 'css':
                if (CDN1_Settings::get('enable_css_versioning')) {
                    $version = CDN1_Settings::get('css_version_string');
                }
                break;
            case 'js':
                if (CDN1_Settings::get('enable_js_versioning')) {
                    $version = CDN1_Settings::get('js_version_string');
                }
                break;
            case 'frontend':
                // Frontend URL doesn't need versioning typically
                break;
        }
        
        // If no version, return URL as-is
        if (empty($version)) {
            return $url;
        }
        
        $parsed = @wp_parse_url($url);
        if (!$parsed) {
            return $url;
        }
        
        // Parse existing query string
        $query_params = [];
        if (!empty($parsed['query'])) {
            parse_str($parsed['query'], $query_params);
        }
        
        // More aggressive version parameter detection - remove ALL known version params
        $version_params = ['v', 'ver', 'version', '_v', '_', 'rev', 'r', '1cdnver'];
        foreach ($version_params as $param) {
            if (isset($query_params[$param])) {
                unset($query_params[$param]);
            }
        }
        
        // Add version
        $query_params['1cdnver'] = $version;
        
        // Rebuild URL
        $base = (isset($parsed['scheme']) ? $parsed['scheme'] . '://' : '//') . 
                (isset($parsed['host']) ? $parsed['host'] : '') .
                (isset($parsed['path']) ? $parsed['path'] : '');
        
        $new_query = http_build_query($query_params);
        $fragment = isset($parsed['fragment']) ? ('#' . $parsed['fragment']) : '';
        
        return $base . ($new_query ? '?' . $new_query : '') . $fragment;
    }
    
    /**
     * Append CDN secure key to URL
     * Only for Default CDN (1cdn.us) URLs
     *
     * @param string $url URL to append key to
     * @return string URL with key appended
     */
    /**
     * Replace srcset URLs
     */
    public function replace_srcset($sources) {
        if (!is_array($sources)) {
            return $sources;
        }
        
        foreach ($sources as &$candidate) {
            if (!empty($candidate['url'])) {
                $candidate['url'] = $this->replace_url($candidate['url']);
            }
        }
        return $sources;
    }
    
    /**
     * Replace image src
     */
    public function replace_image_src($image, $attachment_id, $size, $icon) {
        if (is_array($image) && !empty($image[0])) {
            $image[0] = $this->replace_url($image[0]);
        }
        return $image;
    }
    
    /**
     * Fix style loader tag
     */
    public function fix_style_tag($html, $handle, $href, $media) {
        $fixed = $this->replace_url($href);
        if ($fixed && $fixed !== $href) {
            $html = str_replace($href, esc_url($fixed), $html);
        }
        return $html;
    }
    
    /**
     * Fix script loader tag
     */
    public function fix_script_tag($tag, $handle, $src) {
        $fixed = $this->replace_url($src);
        if ($fixed && $fixed !== $src) {
            $tag = str_replace($src, esc_url($fixed), $tag);
        }
        return $tag;
    }
    
    /**
     * Replace URLs in HTML content
     * Based on custom_cdn_replace_in_html from original functions.php
     */
    public function replace_in_html($html) {
        if (!is_string($html) || $html === '') {
            return $html;
        }
        
        // If Default CDN is enabled, use Default CDN replacement logic
        if (CDN1_Settings::is_default_cdn_enabled()) {
            return $this->replace_in_html_default_cdn($html);
        }
        
        $origin = $this->get_origin();
        $bases = CDN1_Settings::get_cdn_bases();
        $hosts = CDN1_Settings::get_source_hosts();
        $scheme = is_ssl() ? 'https://' : 'http://';
        
        // ---------- 1) Fast absolute replacements (safe, path-scoped) ----------
        $replacements = [];
        $static_dirs = [
            '/wp-content/themes/',
            '/wp-content/plugins/',
            '/wp-includes/',
            '/wp-content/et-cache/'  // ✅ cover Divi unified CSS
        ];
        
        // Frontend URL blanket replacement - only if no specific CDNs are configured
        // If specific CDNs are configured, they will handle routing instead
        $has_specific_cdns = (CDN1_Settings::get('enable_image_cdn') && $bases['img'])
                          || (CDN1_Settings::get('enable_media_cdn') && $bases['media'])
                          || (CDN1_Settings::get('enable_static_cdn') && $bases['static'])
                          || (CDN1_Settings::get('enable_fonts_cdn') && $bases['fonts']);
        
        if (!$has_specific_cdns) {
            foreach ($hosts as $h) {
                $base = $scheme . $h;
                
                // Frontend URL only used when no specific CDNs are configured
                if (CDN1_Settings::get('enable_frontend_url') && $bases['frontend']) {
                    $replacements[$base] = $bases['frontend'];
                }
            }
        }
        
        // Now do regex-based replacements for proper routing
        if (CDN1_Settings::get('enable_image_cdn') && $bases['img']) {
            $html = $this->replace_images_in_html($html, $bases['img']);
        }
        
        if (CDN1_Settings::get('enable_media_cdn') && $bases['media']) {
            $html = $this->replace_media_in_html($html, $bases['media']);
        }
        
        // CSS CDN
        if (CDN1_Settings::get('enable_css_cdn') && $bases['css']) {
            $html = $this->replace_css_in_html($html, $bases['css']);
        }
        
        // JavaScript CDN
        if (CDN1_Settings::get('enable_js_cdn') && $bases['js']) {
            $html = $this->replace_js_in_html($html, $bases['js']);
        }
        
        // Legacy: Static CDN (for backward compatibility)
        if (CDN1_Settings::get('enable_static_cdn') && $bases['static']) {
            // Only apply if CSS/JS CDNs are not individually configured
            if (!CDN1_Settings::get('enable_css_cdn') && !CDN1_Settings::get('enable_js_cdn')) {
                $html = $this->replace_static_in_html($html, $bases['static']);
            }
        }
        
        if (CDN1_Settings::get('enable_fonts_cdn') && $bases['fonts']) {
            $html = $this->replace_fonts_in_html($html, $bases['fonts']);
        }
        
        // Fallback: If Frontend URL is enabled but specific CDN types are not,
        // process those file types through frontend URL with proper versioning
        if (CDN1_Settings::get('enable_frontend_url') && $bases['frontend']) {
            // Process CSS through frontend URL if CSS CDN is not enabled
            if (!CDN1_Settings::get('enable_css_cdn') || empty($bases['css'])) {
                $html = $this->replace_css_in_html($html, $bases['frontend']);
            }
            // Process JS through frontend URL if JS CDN is not enabled
            if (!CDN1_Settings::get('enable_js_cdn') || empty($bases['js'])) {
                $html = $this->replace_js_in_html($html, $bases['frontend']);
            }
            // Process media files through frontend URL if Media CDN is not enabled
            if (!CDN1_Settings::get('enable_media_cdn') || empty($bases['media'])) {
                $html = $this->replace_media_in_html($html, $bases['frontend']);
            }
            // Process fonts through frontend URL if Fonts CDN is not enabled
            if (!CDN1_Settings::get('enable_fonts_cdn') || empty($bases['fonts'])) {
                $html = $this->replace_fonts_in_html($html, $bases['frontend']);
            }
        }
        
        foreach ($replacements as $from => $to) {
            $html = str_replace($from, $to, $html);
        }
        
        // ---------- 2) Fix protocol-relative href/src="//host/..." ----------
        if ($hosts) {
            $host_alt = implode('|', array_map(fn($h) => preg_quote($h, '/'), $hosts));
            
            // Convert only attribute values; then run through CDN replacer
            $html = preg_replace_callback(
                '/\b((?:href|src)\s*=\s*["\'])\/\/(' . $host_alt . ')\/([^"\']+)(["\'])/i',
                function($m) use ($scheme) {
                    $abs = $scheme . $m[2] . '/' . ltrim($m[3], '/');
                    $cdn = $this->replace_url($abs);
                    return $m[1] . esc_url($cdn) . $m[4];
                },
                $html
            );
        }
        
        // ---------- 3) Repair malformed "https:/..." in href/src (single slash) ----------
        $html = preg_replace_callback(
            '/\b((?:href|src)\s*=\s*["\'])https:\/(?!\/)([^"\']+)(["\'])/i',
            function($m) {
                // Normalize to https:// and route via replacer
                $abs = 'https://' . ltrim($m[2], '/');
                $cdn = $this->replace_url($abs);
                return $m[1] . esc_url($cdn) . $m[3];
            },
            $html
        );
        
        // ---------- 4) Inline CSS url(...) absolute hosts → CDN ----------
        if ($hosts && CDN1_Settings::get('enable_static_cdn') && $bases['static']) {
            $host_alt = implode('|', array_map(fn($h) => preg_quote($h, '/'), $hosts));
            $html = preg_replace_callback(
                '/url\((["\']?)(https?:)?\/\/(' . $host_alt . ')\/([^"\')]+)(["\']?)\)/i',
                function ($m) {
                    $proto = $m[2] ?: (is_ssl() ? 'https:' : 'http:');
                    $abs   = $proto . '//' . $m[3] . '/' . ltrim($m[4], '/');
                    $cdn   = $this->replace_url($abs);
                    return 'url(' . $m[1] . $cdn . $m[5] . ')';
                },
                $html
            );
        }
        
        return $html;
    }
    
    /**
     * Replace URLs in HTML content for Default CDN
     */
    private function replace_in_html_default_cdn($html) {
        // Check if Default CDN is enabled at all
        if (!CDN1_Settings::is_default_cdn_enabled()) {
            return $html;
        }
        
        // Prevent multiple processing passes
        if (self::$html_processed) {
            return $html;
        }
        
        $hosts = CDN1_Settings::get_source_hosts();
        if (empty($hosts)) {
            return $html;
        }
        
        // Exclude Default CDN subdomains from source hosts to prevent double wrapping
        $default_cdn_subdomains = ['js.1cdn.us', 'cs.1cdn.us', 'im.1cdn.us', 'ft.1cdn.us', 'mi.1cdn.us'];
        $hosts = array_filter($hosts, function($host) use ($default_cdn_subdomains) {
            return !in_array(strtolower($host), $default_cdn_subdomains);
        });
        
        if (empty($hosts)) {
            return $html;
        }
        
        $scheme = is_ssl() ? 'https://' : 'http://';
        $host_pattern = implode('|', array_map(fn($h) => preg_quote($h, '/'), $hosts));
        
        // Replace JavaScript files
        if (CDN1_Settings::is_default_cdn_type_enabled('js')) {
            $subdomain = CDN1_Settings::get_default_cdn_subdomain('js');
            $html = preg_replace_callback(
                '/(https?:)?\/\/(' . $host_pattern . ')(\/[^"\'\s<>?]*\.js)(\?[^"\'\s<>]*)?/i',
                function($matches) use ($scheme, $subdomain) {
                    $full_url = 'https://' . $matches[2] . $matches[3] . ($matches[4] ?? '');
                    $new_url = $scheme . $subdomain . '/' . $full_url;
                    return $this->add_versioning($new_url, 'js');
                },
                $html
            );
        }
        
        // Replace CSS files
        if (CDN1_Settings::is_default_cdn_type_enabled('css')) {
            $subdomain = CDN1_Settings::get_default_cdn_subdomain('css');
            $html = preg_replace_callback(
                '/(https?:)?\/\/(' . $host_pattern . ')(\/[^"\'\s<>?]*\.css)(\?[^"\'\s<>]*)?/i',
                function($matches) use ($scheme, $subdomain) {
                    $full_url = 'https://' . $matches[2] . $matches[3] . ($matches[4] ?? '');
                    $new_url = $scheme . $subdomain . '/' . $full_url;
                    return $this->add_versioning($new_url, 'css');
                },
                $html
            );
        }
        
        // Replace Images
        if (CDN1_Settings::is_default_cdn_type_enabled('image')) {
            $subdomain = CDN1_Settings::get_default_cdn_subdomain('image');
            $image_ext = '\.(jpg|jpeg|png|gif|svg|webp|ico|bmp|tiff|tif|avif)(\?[^"\'\s<>]*)?';
            $html = preg_replace_callback(
                '/(https?:)?\/\/(' . $host_pattern . ')(\/[^"\'\s<>?]*' . $image_ext . ')/i',
                function($matches) use ($scheme, $subdomain) {
                    $full_url = 'https://' . $matches[2] . $matches[3];
                    $new_url = $scheme . $subdomain . '/' . $full_url;
                    return $this->add_versioning($new_url, 'image');
                },
                $html
            );
        }
        
        // Replace Media files (video/audio)
        if (CDN1_Settings::is_default_cdn_type_enabled('media')) {
            $subdomain = CDN1_Settings::get_default_cdn_subdomain('media');
            $media_ext = '\.(mp4|webm|ogg|ogv|avi|mov|wmv|flv|mkv|m4v|mp3|wav|m4a|aac|flac|oga)(\?[^"\'\s<>]*)?';
            $html = preg_replace_callback(
                '/(https?:)?\/\/(' . $host_pattern . ')(\/[^"\'\s<>?]*' . $media_ext . ')/i',
                function($matches) use ($scheme, $subdomain) {
                    $full_url = 'https://' . $matches[2] . $matches[3];
                    $new_url = $scheme . $subdomain . '/' . $full_url;
                    return $this->add_versioning($new_url, 'media');
                },
                $html
            );
        }
        
        // Replace Fonts
        if (CDN1_Settings::is_default_cdn_type_enabled('fonts')) {
            $subdomain = CDN1_Settings::get_default_cdn_subdomain('fonts');
            $fonts_ext = '\.(woff|woff2|ttf|otf|eot|pdf|doc|docx|xls|xlsx|ppt|pptx|zip|rar|7z|tar|gz|txt|csv)(\?[^"\'\s<>]*)?';
            $html = preg_replace_callback(
                '/(https?:)?\/\/(' . $host_pattern . ')(\/[^"\'\s<>?]*' . $fonts_ext . ')/i',
                function($matches) use ($scheme, $subdomain) {
                    $full_url = 'https://' . $matches[2] . $matches[3];
                    $new_url = $scheme . $subdomain . '/' . $full_url;
                    return $this->add_versioning($new_url, 'fonts');
                },
                $html
            );
        }
        
        // Handle inline CSS url() references
        if (CDN1_Settings::is_default_cdn_type_enabled('css') || 
            CDN1_Settings::is_default_cdn_type_enabled('image') || 
            CDN1_Settings::is_default_cdn_type_enabled('fonts')) {
            
            $html = preg_replace_callback(
                '/url\((["\']?)(https?:)?\/\/(' . $host_pattern . ')\/([^"\')]+)(["\']?)\)/i',
                function ($m) use ($scheme) {
                    $full_url = 'https://' . $m[3] . '/' . $m[4];
                    $path = '/' . $m[4];
                    
                    // Determine asset type
                    $asset_type = null;
                    $subdomain = null;
                    
                    if ($this->is_css_file($path) && CDN1_Settings::is_default_cdn_type_enabled('css')) {
                        $asset_type = 'css';
                        $subdomain = CDN1_Settings::get_default_cdn_subdomain('css');
                    } elseif ($this->is_image_file($path) && CDN1_Settings::is_default_cdn_type_enabled('image')) {
                        $asset_type = 'image';
                        $subdomain = CDN1_Settings::get_default_cdn_subdomain('image');
                    } elseif ($this->is_fonts_file($path) && CDN1_Settings::is_default_cdn_type_enabled('fonts')) {
                        $asset_type = 'fonts';
                        $subdomain = CDN1_Settings::get_default_cdn_subdomain('fonts');
                    }
                    
                    if ($asset_type && $subdomain) {
                        $new_url = $scheme . $subdomain . '/' . $full_url;
                        $new_url = $this->add_versioning($new_url, $asset_type);
                        return 'url(' . $m[1] . $new_url . $m[5] . ')';
                    }
                    
                    return $m[0];
                },
                $html
            );
        }
        
        // Mark HTML as processed to prevent re-processing
        self::$html_processed = true;
        
        return $html;
    }
    
    
    /**
     * Replace images in HTML with Image CDN
     */
    private function replace_images_in_html($html, $image_cdn_base) {
        $hosts = CDN1_Settings::get_source_hosts();
        if (empty($hosts)) {
            return $html;
        }
        
        $host_pattern = implode('|', array_map(fn($h) => preg_quote($h, '/'), $hosts));
        
        // Match image extensions (with optional query parameters)
        // IMPORTANT: Only match actual image files, NOT media files (mp4, webm, etc.)
        $image_ext = '\.(jpg|jpeg|png|gif|svg|webp|ico|bmp|tiff|tif|avif)(\?[^"\'\s<>]*)?';
        
        // Replace absolute image URLs
        $html = preg_replace_callback(
            '/(https?:)?\/\/(' . $host_pattern . ')(\/[^"\'\s<>?]*' . $image_ext . ')/i',
            function($matches) use ($image_cdn_base) {
                $path = $matches[3];
                // Trim whitespace from CDN base before concatenating
                $new_url = rtrim(trim($image_cdn_base), '/') . $path;
                // Add versioning if enabled
                return $this->add_versioning($new_url, 'image');
            },
            $html
        );
        
        return $html;
    }
    
    /**
     * Replace media files (video/audio) in HTML with Media CDN
     */
    private function replace_media_in_html($html, $media_cdn_base) {
        $hosts = CDN1_Settings::get_source_hosts();
        if (empty($hosts)) {
            return $html;
        }
        
        $host_pattern = implode('|', array_map(fn($h) => preg_quote($h, '/'), $hosts));
        
        // Match media extensions (with optional query parameters)
        $media_ext = '\.(mp4|webm|ogg|ogv|avi|mov|wmv|flv|mkv|m4v|mp3|wav|m4a|aac|flac|oga)(\?[^"\'\s<>]*)?';
        
        // Replace absolute media URLs
        $html = preg_replace_callback(
            '/(https?:)?\/\/(' . $host_pattern . ')(\/[^"\'\s<>?]*' . $media_ext . ')/i',
            function($matches) use ($media_cdn_base) {
                $path = $matches[3];
                // Trim whitespace from CDN base before concatenating
                $new_url = rtrim(trim($media_cdn_base), '/') . $path;
                // Add versioning if enabled
                return $this->add_versioning($new_url, 'media');
            },
            $html
        );
        
        return $html;
    }
    
    /**
     * Replace static assets (JS and CSS) in HTML with Static CDN
     */
    private function replace_static_in_html($html, $static_cdn_base) {
        $hosts = CDN1_Settings::get_source_hosts();
        if (empty($hosts)) {
            return $html;
        }
        
        $host_pattern = implode('|', array_map(fn($h) => preg_quote($h, '/'), $hosts));
        
        // Match static asset extensions (with optional query parameters)
        $static_ext = '\.(css|js)(\?[^"\'\s<>]*)?';
        
        // Replace absolute static asset URLs
        $html = preg_replace_callback(
            '/(https?:)?\/\/(' . $host_pattern . ')(\/[^"\'\s<>?]*' . $static_ext . ')/i',
            function($matches) use ($static_cdn_base) {
                $path = $matches[3];
                // Trim whitespace from CDN base before concatenating
                $new_url = rtrim(trim($static_cdn_base), '/') . $path;
                // Detect if CSS or JS and add appropriate versioning
                $cdn_type = (stripos($path, '.css') !== false) ? 'css' : 'js';
                return $this->add_versioning($new_url, $cdn_type);
            },
            $html
        );
        
        return $html;
    }
    
    /**
     * Replace CSS files in HTML with CSS CDN
     */
    private function replace_css_in_html($html, $css_cdn_base) {
        $hosts = CDN1_Settings::get_source_hosts();
        if (empty($hosts)) {
            return $html;
        }
        
        $host_pattern = implode('|', array_map(fn($h) => preg_quote($h, '/'), $hosts));
        $scheme = is_ssl() ? 'https://' : 'http://';
        
        // Match CSS files (with optional query parameters)
        $css_ext = '\.css(\?[^"\'\s<>]*)?';
        
        // Replace absolute CSS URLs
        $html = preg_replace_callback(
            '/(https?:)?\/\/(' . $host_pattern . ')(\/[^"\'\s<>?]*' . $css_ext . ')/i',
            function($matches) use ($css_cdn_base, $scheme) {
                $path = $matches[3];
                // Trim whitespace and trailing slashes from CDN base
                $new_url = rtrim(trim($css_cdn_base), '/') . $path;
                $versioned_url = $this->add_versioning($new_url, 'css');
                // Return with proper scheme
                return $scheme . preg_replace('/^https?:\/\//', '', $versioned_url);
            },
            $html
        );
        
        return $html;
    }
    
    /**
     * Replace JavaScript files in HTML with JS CDN
     */
    private function replace_js_in_html($html, $js_cdn_base) {
        $hosts = CDN1_Settings::get_source_hosts();
        if (empty($hosts)) {
            return $html;
        }
        
        $host_pattern = implode('|', array_map(fn($h) => preg_quote($h, '/'), $hosts));
        $scheme = is_ssl() ? 'https://' : 'http://';
        
        // Match JS files (with optional query parameters)
        $js_ext = '\.js(\?[^"\'\s<>]*)?';
        
        // Replace absolute JS URLs
        $html = preg_replace_callback(
            '/(https?:)?\/\/(' . $host_pattern . ')(\/[^"\'\s<>?]*' . $js_ext . ')/i',
            function($matches) use ($js_cdn_base, $scheme) {
                $path = $matches[3];
                // Trim whitespace and trailing slashes from CDN base
                $new_url = rtrim(trim($js_cdn_base), '/') . $path;
                $versioned_url = $this->add_versioning($new_url, 'js');
                // Return with proper scheme
                return $scheme . preg_replace('/^https?:\/\//', '', $versioned_url);
            },
            $html
        );
        
        return $html;
    }
    
    /**
     * Replace fonts and binary files in HTML with Fonts CDN
     */
    private function replace_fonts_in_html($html, $fonts_cdn_base) {
        $hosts = CDN1_Settings::get_source_hosts();
        if (empty($hosts)) {
            return $html;
        }
        
        $host_pattern = implode('|', array_map(fn($h) => preg_quote($h, '/'), $hosts));
        
        // Match fonts and binary file extensions (with optional query parameters)
        $fonts_ext = '\.(woff|woff2|ttf|otf|eot|pdf|doc|docx|xls|xlsx|ppt|pptx|zip|rar|7z|tar|gz|txt|csv)(\?[^"\'\s<>]*)?';
        
        // Replace absolute fonts/binary URLs
        $html = preg_replace_callback(
            '/(https?:)?\/\/(' . $host_pattern . ')(\/[^"\'\s<>?]*' . $fonts_ext . ')/i',
            function($matches) use ($fonts_cdn_base) {
                $path = $matches[3];
                // Trim whitespace from CDN base before concatenating
                $new_url = rtrim(trim($fonts_cdn_base), '/') . $path;
                // Add versioning if enabled
                return $this->add_versioning($new_url, 'fonts');
            },
            $html
        );
        
        return $html;
    }

    /**
     * Start output buffer
     */
    public function start_output_buffer() {
        if (!is_admin()) {
            ob_start([$this, 'replace_in_html']);
        }
    }
}
