<?php
/**
 * CDN Security and Domain Verification Handler
 *
 * @package CDN1_Manager
 * @since 1.7.0
 */

if (!defined('ABSPATH')) {
    exit;
}

class CDN1_Security {
    
    /**
     * Cloudflare API endpoint for registration
     */
    const API_ENDPOINT = 'https://register.1cdn.us';
    
    /**
     * Initialize security features
     */
    public static function init() {
        // Add verification meta tag to head if auto-inject is enabled
        add_action('wp_head', [__CLASS__, 'inject_verification_meta'], 1);
        
        // Add plugin identification meta tag (always)
        add_action('wp_head', [__CLASS__, 'inject_plugin_meta'], 1);
    }
    
    /**
     * Generate verification token
     *
     * @return string Random verification token
     */
    public static function generate_verification_token() {
        return bin2hex(random_bytes(32)); // 64 character hex string
    }
    
    /**
     * Get or generate verification token
     *
     * @return string Verification token
     */
    public static function get_verification_token() {
        $token = get_option('cdn1_verification_token');
        
        if (empty($token)) {
            $token = self::generate_verification_token();
            update_option('cdn1_verification_token', $token);
        }
        
        return $token;
    }
    
    /**
     * Get master domain from WordPress site URL
     *
     * @return string Master domain (e.g., utility-server.com)
     */
    public static function get_master_domain() {
        $site_url = get_option('siteurl');
        $parsed = parse_url($site_url);
        
        if (!isset($parsed['host'])) {
            return '';
        }
        
        $host = $parsed['host'];
        
        // Remove www. prefix if present
        $host = preg_replace('/^www\./', '', $host);
        
        return $host;
    }
    
    /**
     * Get CDN secure key (stored after successful verification)
     *
     * @return string|false CDN key or false if not registered
     */
    public static function get_cdn_key() {
        return get_option('cdn1_secure_key', false);
    }
    
    /**
     * Check if domain is registered
     *
     * @return bool True if verified and registered
     */
    public static function is_registered() {
        return (bool) get_option('cdn1_registered', false);
    }
    
    /**
     * Inject verification meta tag to head
     */
    public static function inject_verification_meta() {
        $auto_inject = get_option('cdn1_auto_inject_meta', 0);
        
        if (!$auto_inject) {
            return;
        }
        
        $token = self::get_verification_token();
        echo '<meta name="1cdn-verify" content="' . esc_attr($token) . '">' . "\n";
    }
    
    /**
     * Inject plugin identification meta tag to head
     * Only when CDN is enabled
     */
    public static function inject_plugin_meta() {
        // Only inject meta tag when Default CDN is enabled
        $default_cdn_enabled = CDN1_Settings::get('enable_default_cdn', 0);
        if (!$default_cdn_enabled) {
            return;
        }
        
        // Get plugin info
        $version = defined('CDN1_VERSION') ? CDN1_VERSION : '1.7.0';
        
        // Get configuration status
        $is_registered = self::is_registered();
        
        // Get installation ID (generate if doesn't exist)
        $install_id = get_option('cdn1_install_id');
        if (empty($install_id)) {
            $install_id = self::generate_install_id();
            update_option('cdn1_install_id', $install_id);
        }
        
        // Build meta content
        $meta_parts = [
            'v' . $version,
            'cdn:' . ($default_cdn_enabled ? 'on' : 'off'),
            'sec:' . ($is_registered ? 'yes' : 'no'),
            'id:' . substr($install_id, 0, 8), // First 8 chars of install ID
        ];
        
        $meta_content = implode('|', $meta_parts);
        
        // Output meta tag
        echo '<meta name="1cdn-manager" content="' . esc_attr($meta_content) . '">' . "\n";
        
        // Also output as generator tag for better visibility
        echo '<meta name="generator" content="1 CDN Manager ' . esc_attr($version) . '">' . "\n";
    }
    
    /**
     * Generate unique installation ID
     * 
     * @return string Unique installation identifier
     */
    private static function generate_install_id() {
        // Combine site URL, current time, and random bytes for uniqueness
        $site_url = get_option('siteurl');
        $timestamp = time();
        $random = bin2hex(random_bytes(16));
        
        // Create hash of combined data
        $data = $site_url . '|' . $timestamp . '|' . $random;
        return hash('sha256', $data);
    }
    
    /**

     * Get verification file content
     *
     * @return string Content for 1cdn-secure.txt file
     */
    public static function get_verification_file_content() {
        $token = self::get_verification_token();
        return '1cdn-verify=' . $token;
    }
    
    /**
     * Get DNS TXT record value
     *
     * @return string TXT record value
     */
    public static function get_dns_txt_value() {
        $token = self::get_verification_token();
        return '1cdn-verify=' . $token;
    }
    
    /**
     * Initiate domain verification (local check)
     *
     * @param string $method Verification method: 'dns', 'meta', or 'file'
     * @return array Result with success status and message
     */
    public static function verify_domain($method) {
        $domain = self::get_master_domain();
        $token = self::get_verification_token();
        
        if (empty($domain)) {
            return [
                'success' => false,
                'message' => 'Unable to determine site domain.'
            ];
        }
        
        $verified = false;
        $message = '';
        
        switch ($method) {
            case 'dns':
                // Check DNS TXT record
                $verified = self::check_dns_verification($domain, $token);
                $message = $verified ? 'DNS TXT record verified successfully!' : 'DNS TXT record not found. Please wait 5-10 minutes for DNS propagation.';
                break;
                
            case 'meta':
                // Check meta tag
                $verified = self::check_meta_verification($domain, $token);
                $message = $verified ? 'Meta tag verified successfully!' : 'Meta tag not found in page source. Please ensure it\'s in the <head> section.';
                break;
                
            case 'file':
                // Check verification file
                $verified = self::check_file_verification($domain, $token);
                $message = $verified ? 'Verification file found successfully!' : 'Verification file not found. Please upload it to your site root.';
                break;
                
            default:
                return [
                    'success' => false,
                    'message' => 'Invalid verification method.'
                ];
        }
        
        if ($verified) {
            // Store verification method
            update_option('cdn1_verification_method', $method);
            update_option('cdn1_verification_verified_at', time());
        }
        
        return [
            'success' => $verified,
            'message' => $message,
            'details' => [
                'method' => $method,
                'domain' => $domain
            ]
        ];
    }
    
    /**
     * Check DNS TXT verification
     */
    private static function check_dns_verification($domain, $token) {
        $expected_value = '1cdn-verify=' . $token;
        $txt_record = '_1cdn-verification.' . $domain;
        
        // Get DNS TXT records
        $records = @dns_get_record($txt_record, DNS_TXT);
        
        if ($records) {
            foreach ($records as $record) {
                if (isset($record['txt']) && $record['txt'] === $expected_value) {
                    return true;
                }
            }
        }
        
        return false;
    }
    
    /**
     * Check meta tag verification
     */
    private static function check_meta_verification($domain, $token) {
        $url = 'https://' . $domain;
        
        $response = wp_remote_get($url, [
            'timeout' => 15,
            'sslverify' => false
        ]);
        
        if (is_wp_error($response)) {
            return false;
        }
        
        $html = wp_remote_retrieve_body($response);
        
        // Check for meta tag
        $pattern = '/<meta\s+name=["\']1cdn-verify["\']\s+content=["\'](.*?)["\']/i';
        if (preg_match($pattern, $html, $matches)) {
            return $matches[1] === $token;
        }
        
        return false;
    }
    
    /**
     * Check file verification
     */
    private static function check_file_verification($domain, $token) {
        $url = 'https://' . $domain . '/1cdn-verification.txt';
        
        $response = wp_remote_get($url, [
            'timeout' => 15,
            'sslverify' => false
        ]);
        
        if (is_wp_error($response)) {
            return false;
        }
        
        $content = trim(wp_remote_retrieve_body($response));
        
        return $content === $token;
    }
    
    /**
     * Register domain with Cloudflare (includes verification token)
     *
     * @return array Result with success status and message
     */
    public static function register_domain() {
        $domain = self::get_master_domain();
        $method = get_option('cdn1_verification_method', '');
        $token = self::get_verification_token();
        
        if (empty($domain)) {
            return [
                'success' => false,
                'message' => 'Unable to determine site domain.'
            ];
        }
        
        if (empty($method)) {
            return [
                'success' => false,
                'message' => 'Please verify your domain first before registering.'
            ];
        }
        
        if (empty($token)) {
            return [
                'success' => false,
                'message' => 'Verification token not found. Please verify domain again.'
            ];
        }
        
        // Call Cloudflare Registration API (api.1cdn.us)
        // Worker will re-verify domain ownership before registering
        $response = wp_remote_post('https://api.1cdn.us', [
            'timeout' => 30,
            'body' => json_encode([
                'action' => 'register',
                'domain' => $domain,
                'verification_method' => $method,
                'token' => $token
            ]),
            'headers' => [
                'Content-Type' => 'application/json'
            ]
        ]);
        
        if (is_wp_error($response)) {
            return [
                'success' => false,
                'message' => 'Connection error: ' . $response->get_error_message()
            ];
        }
        
        $status_code = wp_remote_retrieve_response_code($response);
        $body = wp_remote_retrieve_body($response);
        
        error_log('Registration API Status: ' . $status_code);
        error_log('Registration API Response: ' . $body);
        
        if ($status_code !== 200) {
            return [
                'success' => false,
                'message' => 'API returned status ' . $status_code . '. Response: ' . substr($body, 0, 200)
            ];
        }
        
        $data = json_decode($body, true);
        
        // Log the actual response for debugging
        error_log('Registration API Response: ' . $body);
        
        if (!$data) {
            return [
                'success' => false,
                'message' => 'Invalid JSON response from API. Response: ' . substr($body, 0, 200)
            ];
        }
        
        if (!isset($data['success'])) {
            return [
                'success' => false,
                'message' => 'API response missing success field. Response: ' . substr($body, 0, 200)
            ];
        }
        
        // If successful, mark as registered
        if ($data['success']) {
            update_option('cdn1_registered', true);
            update_option('cdn1_registered_at', current_time('mysql'));
            update_option('cdn1_is_active', true);
            
            return [
                'success' => true,
                'message' => 'Domain registered successfully! You now have unlimited CDN access.',
                'domain' => $domain,
                'verification_method' => $method
            ];
        }
        
        return [
            'success' => false,
            'message' => isset($data['message']) ? $data['message'] : 'Registration failed',
            'error' => isset($data['error']) ? $data['error'] : null
        ];
    }
    
    /**
     * Regenerate verification token
     *
     * @return string New verification token
     */
    public static function regenerate_token() {
        delete_option('cdn1_verification_token');
        return self::get_verification_token();
    }
    
    /**
     * Revoke registration (soft delete on Cloudflare, requires verification)
     * Also automatically disables Default CDN if enabled
     */
    public static function revoke_registration() {
        $domain = self::get_master_domain();
        $method = get_option('cdn1_verification_method', '');
        $token = self::get_verification_token();
        
        // Call Cloudflare API to revoke (soft delete)
        // Worker will verify ownership before allowing revocation
        if (!empty($domain) && !empty($method) && !empty($token)) {
            $response = wp_remote_post('https://api.1cdn.us', [
                'timeout' => 30,
                'body' => json_encode([
                    'action' => 'revoke',
                    'domain' => $domain,
                    'verification_method' => $method,
                    'token' => $token
                ]),
                'headers' => [
                    'Content-Type' => 'application/json'
                ]
            ]);
        }
        
        // Automatically disable Default CDN if enabled
        $default_cdn_enabled = CDN1_Settings::get('enable_default_cdn', 0);
        if ($default_cdn_enabled) {
            CDN1_Settings::set('enable_default_cdn', 0);
            // Also disable all Default CDN asset types
            CDN1_Settings::set('enable_default_cdn_js', 0);
            CDN1_Settings::set('enable_default_cdn_css', 0);
            CDN1_Settings::set('enable_default_cdn_images', 0);
            CDN1_Settings::set('enable_default_cdn_fonts', 0);
            CDN1_Settings::set('enable_default_cdn_media', 0);
        }
        
        // Clear local registration data regardless of API response
        delete_option('cdn1_registered');
        delete_option('cdn1_registered_at');
        delete_option('cdn1_is_active');
        delete_option('cdn1_verification_method');
        delete_option('cdn1_verification_verified_at');
        delete_option('cdn1_verification_token');
        delete_option('cdn1_auto_inject_meta');
    }
    
    /**
     * Get registration information from Cloudflare KV
     */
    public static function get_registration_info() {
        $domain = self::get_master_domain();
        
        if (empty($domain)) {
            return null;
        }
        
        // Call Cloudflare API to check registration
        $response = wp_remote_post('https://api.1cdn.us', [
            'timeout' => 15,
            'body' => json_encode([
                'action' => 'check',
                'domain' => $domain
            ]),
            'headers' => [
                'Content-Type' => 'application/json'
            ]
        ]);
        
        if (is_wp_error($response)) {
            return null;
        }
        
        $body = wp_remote_retrieve_body($response);
        $data = json_decode($body, true);
        
        if ($data && isset($data['is_active']) && $data['is_active']) {
            return $data;
        }
        
        return null;
    }
    /**
     * Get registration status details
     */
    public static function get_status() {
        $is_registered = self::is_registered();
        $domain = self::get_master_domain();
        $token = self::get_verification_token();
        
        // If registered, fetch live info from Cloudflare
        $registration_info = null;
        if ($is_registered) {
            $registration_info = self::get_registration_info();
        }
        
        return [
            'registered' => $is_registered,
            'domain' => $domain,
            'verification_token' => $token,
            'registration_info' => $registration_info
        ];
    }
}

// Initialize
CDN1_Security::init();
