<?php
/**
 * CDN Settings Management
 * Handles all plugin settings and options
 */

if (!defined('ABSPATH')) {
    exit;
}

class CDN1_Settings {
    
    private static $instance = null;
    
    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    private function __construct() {
        add_action('admin_init', [$this, 'register_settings']);
    }
    
    public function register_settings() {
        // Default CDN Settings (predefined 1cdn.us subdomains)
        register_setting('cdn1_settings', 'cdn1_enable_default_cdn', [
            'type' => 'boolean',
            'default' => 0,
            'sanitize_callback' => [$this, 'sanitize_boolean']
        ]);
        
        register_setting('cdn1_settings', 'cdn1_enable_default_cdn_js', [
            'type' => 'boolean',
            'default' => 0,
            'sanitize_callback' => [$this, 'sanitize_boolean']
        ]);
        
        register_setting('cdn1_settings', 'cdn1_enable_default_cdn_css', [
            'type' => 'boolean',
            'default' => 0,
            'sanitize_callback' => [$this, 'sanitize_boolean']
        ]);
        
        register_setting('cdn1_settings', 'cdn1_enable_default_cdn_images', [
            'type' => 'boolean',
            'default' => 0,
            'sanitize_callback' => [$this, 'sanitize_boolean']
        ]);
        
        register_setting('cdn1_settings', 'cdn1_enable_default_cdn_fonts', [
            'type' => 'boolean',
            'default' => 0,
            'sanitize_callback' => [$this, 'sanitize_boolean']
        ]);
        
        register_setting('cdn1_settings', 'cdn1_enable_default_cdn_media', [
            'type' => 'boolean',
            'default' => 0,
            'sanitize_callback' => [$this, 'sanitize_boolean']
        ]);
        
        // Origin URL (where CDN pulls from)
        register_setting('cdn1_settings', 'cdn1_enable_origin_url', [
            'type' => 'boolean',
            'default' => 0,
            'sanitize_callback' => [$this, 'sanitize_boolean']
        ]);
        
        register_setting('cdn1_settings', 'cdn1_origin_url', [
            'type' => 'string',
            'default' => '',
            'sanitize_callback' => 'esc_url_raw'
        ]);
        
        // Admin URL (separate admin domain)
        register_setting('cdn1_settings', 'cdn1_enable_admin_url', [
            'type' => 'boolean',
            'default' => 0,
            'sanitize_callback' => [$this, 'sanitize_boolean']
        ]);
        
        register_setting('cdn1_settings', 'cdn1_admin_url', [
            'type' => 'string',
            'default' => '',
            'sanitize_callback' => 'esc_url_raw'
        ]);
        
        // Image CDN Settings
        register_setting('cdn1_settings', 'cdn1_enable_image_cdn', [
            'type' => 'boolean',
            'default' => 0,
            'sanitize_callback' => [$this, 'sanitize_boolean']
        ]);
        
        register_setting('cdn1_settings', 'cdn1_cdn_image_base', [
            'type' => 'string',
            'default' => '',
            'sanitize_callback' => 'esc_url_raw'
        ]);
        
        // Media CDN Settings (Video/Audio)
        register_setting('cdn1_settings', 'cdn1_enable_media_cdn', [
            'type' => 'boolean',
            'default' => 0,
            'sanitize_callback' => [$this, 'sanitize_boolean']
        ]);
        
        register_setting('cdn1_settings', 'cdn1_cdn_media_base', [
            'type' => 'string',
            'default' => '',
            'sanitize_callback' => 'esc_url_raw'
        ]);
        
        // CSS CDN Settings
        register_setting('cdn1_settings', 'cdn1_enable_css_cdn', [
            'type' => 'boolean',
            'default' => 0,
            'sanitize_callback' => [$this, 'sanitize_boolean']
        ]);
        
        register_setting('cdn1_settings', 'cdn1_cdn_css_base', [
            'type' => 'string',
            'default' => '',
            'sanitize_callback' => 'esc_url_raw'
        ]);
        
        // JavaScript CDN Settings
        register_setting('cdn1_settings', 'cdn1_enable_js_cdn', [
            'type' => 'boolean',
            'default' => 0,
            'sanitize_callback' => [$this, 'sanitize_boolean']
        ]);
        
        register_setting('cdn1_settings', 'cdn1_cdn_js_base', [
            'type' => 'string',
            'default' => '',
            'sanitize_callback' => 'esc_url_raw'
        ]);
        
        // Legacy Static CDN Settings (kept for backward compatibility)
        register_setting('cdn1_settings', 'cdn1_enable_static_cdn', [
            'type' => 'boolean',
            'default' => 0,
            'sanitize_callback' => [$this, 'sanitize_boolean']
        ]);
        
        register_setting('cdn1_settings', 'cdn1_cdn_static_base', [
            'type' => 'string',
            'default' => '',
            'sanitize_callback' => 'esc_url_raw'
        ]);
        
        // Fonts CDN Settings (Fonts/PDFs/Binary)
        register_setting('cdn1_settings', 'cdn1_enable_fonts_cdn', [
            'type' => 'boolean',
            'default' => 0,
            'sanitize_callback' => [$this, 'sanitize_boolean']
        ]);
        
        register_setting('cdn1_settings', 'cdn1_cdn_fonts_base', [
            'type' => 'string',
            'default' => '',
            'sanitize_callback' => 'esc_url_raw'
        ]);
        
        // Frontend URL Settings (Domain Replacement)
        register_setting('cdn1_settings', 'cdn1_enable_frontend_url', [
            'type' => 'boolean',
            'default' => 0,
            'sanitize_callback' => [$this, 'sanitize_boolean']
        ]);
        
        register_setting('cdn1_settings', 'cdn1_cdn_frontend_url', [
            'type' => 'string',
            'default' => '',
            'sanitize_callback' => 'esc_url_raw'
        ]);
        
        // Advanced Settings
        
        register_setting('cdn1_settings', 'cdn1_cache_enabled', [
            'type' => 'boolean',
            'default' => 1,
            'sanitize_callback' => [$this, 'sanitize_boolean']
        ]);
        
        register_setting('cdn1_settings', 'cdn1_custom_source_hosts', [
            'type' => 'string',
            'default' => '',
            'sanitize_callback' => 'sanitize_textarea_field'
        ]);
        
        // Dynamic Content Rewriting
        register_setting('cdn1_settings', 'cdn1_enable_dynamic_rewriting', [
            'type' => 'boolean',
            'default' => 0,
            'sanitize_callback' => [$this, 'sanitize_boolean']
        ]);
        
        register_setting('cdn1_settings', 'cdn1_dynamic_rewriting_auto_rules', [
            'type' => 'boolean',
            'default' => 1,
            'sanitize_callback' => [$this, 'sanitize_boolean']
        ]);
        
        register_setting('cdn1_settings', 'cdn1_dynamic_rewriting_aggressive', [
            'type' => 'boolean',
            'default' => 0,
            'sanitize_callback' => [$this, 'sanitize_boolean']
        ]);
        
        register_setting('cdn1_settings', 'cdn1_dynamic_rewrite_rules', [
            'type' => 'string',
            'default' => '',
            'sanitize_callback' => 'sanitize_textarea_field'
        ]);
        
        // Site URL Configuration Settings
        register_setting('cdn1_settings', 'cdn1_enable_site_url_management', [
            'type' => 'boolean',
            'default' => 0,
            'sanitize_callback' => [$this, 'sanitize_boolean']
        ]);
        
        register_setting('cdn1_settings', 'cdn1_custom_site_url', [
            'type' => 'string',
            'default' => '',
            'sanitize_callback' => [$this, 'sanitize_url']
        ]);
        
        register_setting('cdn1_settings', 'cdn1_custom_admin_url', [
            'type' => 'string',
            'default' => '',
            'sanitize_callback' => [$this, 'sanitize_url']
        ]);
    }
    
    public function sanitize_boolean($value) {
        return (int) !!$value;
    }
    
    public function sanitize_url($value) {
        // Trim whitespace
        $value = trim($value);
        // Sanitize URL
        $value = esc_url_raw($value);
        // Trim again after sanitization
        $value = trim($value);
        // Remove trailing slash
        return rtrim($value, '/');
    }
    
    /**
     * Get a setting value
     */
    public static function get($key, $default = null) {
        $value = get_option('cdn1_' . $key, $default);
        return apply_filters('cdn1_setting_' . $key, $value);
    }
    
    /**
     * Update a setting value
     */
    public static function set($key, $value) {
        return update_option('cdn1_' . $key, $value);
    }
    
    /**
     * Get all CDN bases
     */
    public static function get_cdn_bases() {
        return [
            'img' => rtrim(trim(self::get('cdn_image_base', '')), '/'),
            'media' => rtrim(trim(self::get('cdn_media_base', '')), '/'),
            'css' => rtrim(trim(self::get('cdn_css_base', '')), '/'),
            'js' => rtrim(trim(self::get('cdn_js_base', '')), '/'),
            'static' => rtrim(trim(self::get('cdn_static_base', '')), '/'), // Legacy support
            'fonts' => rtrim(trim(self::get('cdn_fonts_base', '')), '/'),
            'frontend' => rtrim(trim(self::get('cdn_frontend_url', '')), '/'),
        ];
    }
    
    /**
     * Get source hosts
     */
    public static function get_source_hosts() {
        $hosts = [];
        
        // Current request host
        $h1 = $_SERVER['HTTP_HOST'] ?? '';
        if ($h1) $hosts[] = strtolower($h1);
        
        // Home URL host
        $h2 = wp_parse_url(home_url(), PHP_URL_HOST);
        if ($h2) $hosts[] = strtolower($h2);
        
        // Site URL host
        $h3 = wp_parse_url(site_url(), PHP_URL_HOST);
        if ($h3) $hosts[] = strtolower($h3);
        
        // Add CDN domains EXCEPT Frontend URL to allow re-routing
        // This allows re-routing URLs from wrong CDN to correct CDN
        $cdn_bases = self::get_cdn_bases();
        $cdn_types = ['img', 'media', 'static', 'fonts']; // Exclude 'frontend'
        foreach ($cdn_types as $type) {
            if (!empty($cdn_bases[$type])) {
                $cdn_host = wp_parse_url($cdn_bases[$type], PHP_URL_HOST);
                if ($cdn_host) {
                    $hosts[] = strtolower($cdn_host);
                }
            }
        }
        
        // Custom hosts from settings
        $custom = self::get('custom_source_hosts', '');
        if (!empty($custom)) {
            $custom_hosts = array_map('trim', explode("\n", $custom));
            foreach ($custom_hosts as $host) {
                if (!empty($host)) {
                    $hosts[] = strtolower($host);
                }
            }
        }
        
        $hosts = array_values(array_unique(array_filter($hosts)));
        return apply_filters('cdn1_source_hosts', $hosts);
    }
    
    /**
     * Check if setup is completed
     */
    public static function is_setup_completed() {
        return (bool) self::get('setup_completed', 0);
    }
    
    /**
     * Check if Default CDN is enabled
     */
    public static function is_default_cdn_enabled() {
        return (bool) self::get('enable_default_cdn', 0);
    }
    
    /**
     * Get Default CDN subdomain for asset type
     */
    public static function get_default_cdn_subdomain($asset_type) {
        $subdomains = [
            'js' => 'js.1cdn.us',
            'css' => 'cs.1cdn.us',
            'image' => 'im.1cdn.us',
            'fonts' => 'ft.1cdn.us',
            'media' => 'mi.1cdn.us',
        ];
        
        return isset($subdomains[$asset_type]) ? $subdomains[$asset_type] : null;
    }
    
    /**
     * Check if specific Default CDN asset type is enabled
     */
    public static function is_default_cdn_type_enabled($asset_type) {
        if (!self::is_default_cdn_enabled()) {
            return false;
        }
        
        $setting_map = [
            'js' => 'enable_default_cdn_js',
            'css' => 'enable_default_cdn_css',
            'image' => 'enable_default_cdn_images',
            'fonts' => 'enable_default_cdn_fonts',
            'media' => 'enable_default_cdn_media',
        ];
        
        $setting_key = isset($setting_map[$asset_type]) ? $setting_map[$asset_type] : null;
        
        if (!$setting_key) {
            return false;
        }
        
        return (bool) self::get($setting_key, 0);
    }
}
