<?php
/**
 * Site URL Configuration Manager
 * 
 * Handles WordPress site URL configuration to prevent plugins from
 * detecting wrong URLs when admin is accessed from a different domain.
 *
 * @package CDN1_Manager
 * @since 1.4.0
 */

if (!defined('ABSPATH')) {
    exit;
}

class CDN1_Site_Config {
    
    /**
     * Initialize the site configuration manager
     */
    public static function init() {
        // Auto-enable if Admin URL is configured in CDN Settings
        self::auto_enable_if_needed();
        
        // Override URLs if feature is enabled
        if (CDN1_Settings::get('enable_site_url_management')) {
            // Use priority 1 to run very early
            add_filter('option_home', [__CLASS__, 'override_home_url'], 1);
            add_filter('option_siteurl', [__CLASS__, 'override_siteurl'], 1);
            
            // Pre-option filters (even earlier)
            add_filter('pre_option_home', [__CLASS__, 'override_home_url'], 1);
            add_filter('pre_option_siteurl', [__CLASS__, 'override_siteurl'], 1);
            
            // Also filter the actual home_url() and site_url() functions
            add_filter('home_url', [__CLASS__, 'filter_home_url'], 1, 4);
            add_filter('site_url', [__CLASS__, 'filter_site_url'], 1, 4);
        }
    }
    
    /**
     * Auto-enable feature if Admin URL is configured in CDN Settings
     */
    private static function auto_enable_if_needed() {
        // Check if auto-enable has already been processed
        $auto_enable_done = get_option('cdn1_site_config_auto_enable_done', false);
        
        if ($auto_enable_done) {
            return; // Already processed, don't auto-enable again
        }
        
        // Check if Admin URL is configured in CDN Settings
        $admin_url = CDN1_Settings::get('admin_url');
        $frontend_url = CDN1_Settings::get('cdn_frontend_url');
        
        // Auto-enable if Admin URL is configured
        if (!empty($admin_url)) {
            CDN1_Settings::set('enable_site_url_management', 1);
            
            // Auto-fill Site URL from Frontend URL if available
            if (!empty($frontend_url)) {
                CDN1_Settings::set('custom_site_url', rtrim($frontend_url, '/'));
            }
            
            // Auto-fill Admin URL
            CDN1_Settings::set('custom_admin_url', rtrim($admin_url, '/'));
            
            // Mark as done so we don't auto-enable again
            update_option('cdn1_site_config_auto_enable_done', true);
        }
    }
    
    /**
     * Override home URL (WP_HOME)
     */
    public static function override_home_url($url) {
        $custom_url = CDN1_Settings::get('custom_site_url');
        
        if (!empty($custom_url)) {
            return rtrim($custom_url, '/');
        }
        
        return $url;
    }
    
    /**
     * Override site URL (WP_SITEURL) - automatically uses Site URL value
     */
    public static function override_siteurl($url) {
        $custom_url = CDN1_Settings::get('custom_site_url');
        
        if (!empty($custom_url)) {
            return rtrim($custom_url, '/');
        }
        
        return $url;
    }
    
    /**
     * Filter home_url() function calls
     */
    public static function filter_home_url($url, $path = '', $orig_scheme = null, $blog_id = null) {
        $custom_url = CDN1_Settings::get('custom_site_url');
        
        if (empty($custom_url)) {
            return $url;
        }
        
        $custom_url = rtrim($custom_url, '/');
        
        // Simply replace the domain in the URL
        $parsed_original = parse_url($url);
        $parsed_custom = parse_url($custom_url);
        
        if (!$parsed_original || !$parsed_custom) {
            return $url;
        }
        
        // Build the new URL by replacing the domain
        $new_url = $parsed_custom['scheme'] . '://' . $parsed_custom['host'];
        
        // Add port if custom URL has it
        if (isset($parsed_custom['port'])) {
            $new_url .= ':' . $parsed_custom['port'];
        }
        
        // Keep the path from the original URL
        if (!empty($path)) {
            $new_url .= '/' . ltrim($path, '/');
        } elseif (isset($parsed_original['path']) && $parsed_original['path'] !== '/') {
            $new_url .= $parsed_original['path'];
        }
        
        // Keep query and fragment
        if (isset($parsed_original['query'])) {
            $new_url .= '?' . $parsed_original['query'];
        }
        if (isset($parsed_original['fragment'])) {
            $new_url .= '#' . $parsed_original['fragment'];
        }
        
        return $new_url;
    }
    
    /**
     * Filter site_url() function calls
     */
    public static function filter_site_url($url, $path = '', $orig_scheme = null, $blog_id = null) {
        // Use same logic as home_url
        return self::filter_home_url($url, $path, $orig_scheme, $blog_id);
    }
    
    /**
     * Get current environment diagnostics
     */
    public static function get_diagnostics() {
        $diagnostics = [
            'http_host' => $_SERVER['HTTP_HOST'] ?? 'Unknown',
            'request_uri' => $_SERVER['REQUEST_URI'] ?? 'Unknown',
            'server_name' => $_SERVER['SERVER_NAME'] ?? 'Unknown',
            'detected_home_url' => get_option('home'),
            'detected_site_url' => get_option('siteurl'),
            'wp_home_constant' => defined('WP_HOME') ? WP_HOME : 'Not defined',
            'wp_siteurl_constant' => defined('WP_SITEURL') ? WP_SITEURL : 'Not defined',
            'wp_admin_url_constant' => defined('WP_ADMIN_URL') ? WP_ADMIN_URL : 'Not defined',
            'custom_site_url' => CDN1_Settings::get('custom_site_url'),
            'custom_admin_url' => CDN1_Settings::get('custom_admin_url'),
            'feature_enabled' => CDN1_Settings::get('enable_site_url_management'),
            'effective_home_url' => home_url(),
            'effective_site_url' => site_url(),
            'admin_url' => admin_url(),
            'is_ssl' => is_ssl(),
        ];
        
        return $diagnostics;
    }
    
    /**
     * Check if URLs are properly configured
     */
    public static function check_configuration() {
        $issues = [];
        
        $custom_site_url = CDN1_Settings::get('custom_site_url');
        $feature_enabled = CDN1_Settings::get('enable_site_url_management');
        
        if ($feature_enabled) {
            // Check if custom URLs are set
            if (empty($custom_site_url)) {
                $issues[] = [
                    'type' => 'error',
                    'message' => 'Site URL (WP_HOME) is empty. Please configure it.'
                ];
            }
            
            // Check if URLs are valid
            if (!empty($custom_site_url) && !filter_var($custom_site_url, FILTER_VALIDATE_URL)) {
                $issues[] = [
                    'type' => 'error',
                    'message' => 'Site URL is not a valid URL.'
                ];
            }
            
            // Check for conflicts with wp-config.php constants
            if (defined('WP_HOME')) {
                $issues[] = [
                    'type' => 'warning',
                    'message' => 'WP_HOME is defined in wp-config.php (' . WP_HOME . '). Plugin settings may be overridden.'
                ];
            }
            
            if (defined('WP_SITEURL')) {
                $issues[] = [
                    'type' => 'warning',
                    'message' => 'WP_SITEURL is defined in wp-config.php (' . WP_SITEURL . '). Plugin settings may be overridden.'
                ];
            }
            
            if (defined('WP_ADMIN_URL')) {
                $issues[] = [
                    'type' => 'warning',
                    'message' => 'WP_ADMIN_URL is defined in wp-config.php (' . WP_ADMIN_URL . '). This may conflict with plugin settings.'
                ];
            }
            
            // Check if current access is from different domain
            $current_host = $_SERVER['HTTP_HOST'] ?? '';
            $site_host = parse_url($custom_site_url, PHP_URL_HOST);
            
            if ($current_host && $site_host && $current_host !== $site_host) {
                $issues[] = [
                    'type' => 'info',
                    'message' => "Admin accessed from different domain ($current_host) than configured Site URL ($site_host). This is what the feature is designed to handle."
                ];
            }
        }
        
        return $issues;
    }
    
    /**
     * Get list of plugins that might be affected
     */
    public static function get_affected_plugins() {
        $known_plugins = [
            'google-site-kit/google-site-kit.php' => 'Google Site Kit',
            'jetpack/jetpack.php' => 'Jetpack',
            'wordfence/wordfence.php' => 'Wordfence Security',
            'really-simple-ssl/rlrsssl-really-simple-ssl.php' => 'Really Simple SSL',
            'all-in-one-seo-pack/all_in_one_seo_pack.php' => 'All in One SEO',
            'wordpress-seo/wp-seo.php' => 'Yoast SEO',
            'google-analytics-for-wordpress/googleanalytics.php' => 'MonsterInsights',
        ];
        
        $active_plugins = get_option('active_plugins', []);
        $affected = [];
        
        foreach ($known_plugins as $plugin_file => $plugin_name) {
            if (in_array($plugin_file, $active_plugins)) {
                $affected[] = $plugin_name;
            }
        }
        
        return $affected;
    }
    
    /**
     * Auto-detect recommended URLs
     */
    public static function auto_detect_urls() {
        // Get URLs from CDN settings
        $frontend_url = CDN1_Settings::get('cdn_frontend_url');
        $admin_url = CDN1_Settings::get('admin_url');
        
        // Check if respective CDN types are enabled
        $frontend_enabled = CDN1_Settings::get('enable_frontend_url');
        $admin_enabled = CDN1_Settings::get('enable_admin_url');
        
        // Start with WordPress defaults
        $detected = [
            'site_url' => get_option('home'),
            'admin_url' => '',
        ];
        
        // Only use Frontend URL if Frontend CDN is enabled
        if (!empty($frontend_url) && $frontend_enabled) {
            $detected['site_url'] = rtrim($frontend_url, '/');
        }
        
        // Only use Admin URL if Admin CDN is enabled
        if (!empty($admin_url) && $admin_enabled) {
            $detected['admin_url'] = rtrim($admin_url, '/');
        }
        
        // If WP_HOME or WP_ADMIN_URL are defined, use those (they override everything)
        if (defined('WP_HOME')) {
            $detected['site_url'] = WP_HOME;
        }
        
        if (defined('WP_ADMIN_URL')) {
            $detected['admin_url'] = WP_ADMIN_URL;
        }
        
        return $detected;
    }
}
