<?php
/**
 * CDN Manager Update Checker
 * 
 * Handles plugin update checks from S3-hosted update-info.json
 * 
 * @package CDN_Manager
 */

if (!defined('ABSPATH')) {
    exit;
}

class CDN1_Update_Checker {
    
    private static $instance = null;
    private $update_url = 'https://awsmarketplace-public-personal.s3.us-east-1.amazonaws.com/1cdn-wordpress-plugin/update-info.json';
    private $plugin_slug = '1cdn-manager/1cdn-manager.php';
    
    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    private function __construct() {
        // Hook into WordPress update system
        add_filter('pre_set_site_transient_update_plugins', [$this, 'check_for_updates']);
        add_filter('plugins_api', [$this, 'plugin_info'], 20, 3);
    }
    
    /**
     * Check for plugin updates
     */
    public function check_for_updates($transient) {
        if (empty($transient->checked)) {
            return $transient;
        }
        
        // Get remote version info
        $remote = $this->get_remote_version();
        
        if ($remote === false) {
            return $transient;
        }
        
        // Compare versions
        $current_version = CDN1_VERSION;
        
        if (version_compare($current_version, $remote->version, '<')) {
            $plugin_data = [
                'slug' => '1cdn-manager',
                'plugin' => $this->plugin_slug,
                'new_version' => $remote->version,
                'url' => $remote->homepage,
                'package' => $remote->download_url,
                'tested' => $remote->tested,
                'requires' => $remote->requires,
                'requires_php' => $remote->requires_php,
            ];
            
            $transient->response[$this->plugin_slug] = (object) $plugin_data;
        }
        
        return $transient;
    }
    
    /**
     * Get plugin information for the plugin details popup
     */
    public function plugin_info($false, $action, $response) {
        // Only for our plugin
        if ($action !== 'plugin_information') {
            return $false;
        }
        
        if (!isset($response->slug) || $response->slug !== '1cdn-manager') {
            return $false;
        }
        
        // Get remote version info
        $remote = $this->get_remote_version();
        
        if ($remote === false) {
            return $false;
        }
        
        // Build plugin info object
        $plugin_info = new stdClass();
        $plugin_info->name = $remote->name;
        $plugin_info->slug = '1cdn-manager';
        $plugin_info->version = $remote->version;
        $plugin_info->author = '<a href="' . $remote->author_profile . '">' . $remote->author . '</a>';
        $plugin_info->homepage = $remote->homepage;
        $plugin_info->requires = $remote->requires;
        $plugin_info->tested = $remote->tested;
        $plugin_info->requires_php = $remote->requires_php;
        $plugin_info->last_updated = $remote->last_updated;
        $plugin_info->download_link = $remote->download_url;
        
        // Sections
        $plugin_info->sections = [
            'description' => $remote->sections->description,
            'installation' => $remote->sections->installation,
            'changelog' => $remote->sections->changelog,
        ];
        
        if (isset($remote->sections->faq)) {
            $plugin_info->sections['faq'] = $remote->sections->faq;
        }
        
        // Banners and icons
        if (isset($remote->banners)) {
            $plugin_info->banners = [
                'low' => $remote->banners->low ?? '',
                'high' => $remote->banners->high ?? '',
            ];
        }
        
        if (isset($remote->icons)) {
            $plugin_info->icons = [
                '1x' => $remote->icons->{'1x'} ?? '',
                '2x' => $remote->icons->{'2x'} ?? '',
            ];
        }
        
        // Screenshots
        if (isset($remote->screenshots) && is_array($remote->screenshots)) {
            $plugin_info->screenshots = $remote->screenshots;
        }
        
        return $plugin_info;
    }
    
    /**
     * Get remote version information from S3
     */
    private function get_remote_version() {
        // Check cache first (12 hour cache)
        $cache_key = 'cdn1_remote_version_info';
        $cached = get_transient($cache_key);
        
        if ($cached !== false) {
            return $cached;
        }
        
        // Fetch from S3
        $response = wp_remote_get($this->update_url, [
            'timeout' => 10,
            'headers' => [
                'Accept' => 'application/json',
            ],
        ]);
        
        if (is_wp_error($response)) {
            error_log('CDN1 Update Check Error: ' . $response->get_error_message());
            return false;
        }
        
        $response_code = wp_remote_retrieve_response_code($response);
        if ($response_code !== 200) {
            error_log('CDN1 Update Check: HTTP ' . $response_code);
            return false;
        }
        
        $body = wp_remote_retrieve_body($response);
        $data = json_decode($body);
        
        if (json_last_error() !== JSON_ERROR_NONE) {
            error_log('CDN1 Update Check: JSON decode error');
            return false;
        }
        
        // Cache for 12 hours
        set_transient($cache_key, $data, 12 * HOUR_IN_SECONDS);
        
        return $data;
    }
    
    /**
     * Force check for updates (clear cache)
     */
    public static function force_check() {
        delete_transient('cdn1_remote_version_info');
        delete_site_transient('update_plugins');
        
        // Trigger WordPress to check for updates
        wp_update_plugins();
        
        return true;
    }
}
