<?php
/**
 * CDN Analytics
 * Tracks CDN usage and performance metrics
 */

if (!defined('ABSPATH')) {
    exit;
}

class USCDN_Analytics {
    
    private static $instance = null;
    private $stats = [];
    
    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    private function __construct() {
        if (USCDN_Settings::get('analytics_enabled', 1)) {
            add_action('shutdown', [$this, 'save_stats']);
        }
    }
    
    /**
     * Track URL replacement
     */
    public function track_replacement($type, $original_url, $cdn_url) {
        if (!isset($this->stats[$type])) {
            $this->stats[$type] = 0;
        }
        $this->stats[$type]++;
    }
    
    /**
     * Save statistics
     */
    public function save_stats() {
        if (empty($this->stats)) {
            return;
        }
        
        $date = current_time('Y-m-d');
        $existing = get_option('uscdn_analytics_' . $date, []);
        
        foreach ($this->stats as $type => $count) {
            if (!isset($existing[$type])) {
                $existing[$type] = 0;
            }
            $existing[$type] += $count;
        }
        
        update_option('uscdn_analytics_' . $date, $existing, false);
    }
    
    /**
     * Get analytics data for date range
     */
    public static function get_analytics($days = 7) {
        $data = [];
        $current = time();
        
        for ($i = 0; $i < $days; $i++) {
            $date = date('Y-m-d', $current - ($i * DAY_IN_SECONDS));
            $stats = get_option('uscdn_analytics_' . $date, []);
            
            if (!empty($stats)) {
                $data[$date] = $stats;
            }
        }
        
        return $data;
    }
    
    /**
     * Get summary statistics
     */
    public static function get_summary() {
        $analytics = self::get_analytics(30);
        
        $summary = [
            'total_replacements' => 0,
            'image_replacements' => 0,
            'static_replacements' => 0,
            'link_replacements' => 0,
            'cache_hits' => 0
        ];
        
        foreach ($analytics as $date => $stats) {
            foreach ($stats as $type => $count) {
                if (isset($summary[$type])) {
                    $summary[$type] += $count;
                }
                $summary['total_replacements'] += $count;
            }
        }
        
        return $summary;
    }
    
    /**
     * Clear old analytics data
     */
    public static function cleanup_old_data($days_to_keep = 30) {
        global $wpdb;
        
        $cutoff_date = date('Y-m-d', time() - ($days_to_keep * DAY_IN_SECONDS));
        
        $options = $wpdb->get_results(
            $wpdb->prepare(
                "SELECT option_name FROM {$wpdb->options} 
                WHERE option_name LIKE 'uscdn_analytics_%%' 
                AND option_name < %s",
                'uscdn_analytics_' . $cutoff_date
            )
        );
        
        foreach ($options as $option) {
            delete_option($option->option_name);
        }
    }
}
