<?php
/**
 * CDN Settings Management
 * Handles all plugin settings and options
 */

if (!defined('ABSPATH')) {
    exit;
}

class USCDN_Settings {
    
    private static $instance = null;
    
    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    private function __construct() {
        add_action('admin_init', [$this, 'register_settings']);
    }
    
    public function register_settings() {
        // Origin URL (where CDN pulls from)
        register_setting('uscdn_settings', 'uscdn_enable_origin_url', [
            'type' => 'boolean',
            'default' => 0,
            'sanitize_callback' => [$this, 'sanitize_boolean']
        ]);
        
        register_setting('uscdn_settings', 'uscdn_origin_url', [
            'type' => 'string',
            'default' => '',
            'sanitize_callback' => 'esc_url_raw'
        ]);
        
        // Admin URL (separate admin domain)
        register_setting('uscdn_settings', 'uscdn_enable_admin_url', [
            'type' => 'boolean',
            'default' => 0,
            'sanitize_callback' => [$this, 'sanitize_boolean']
        ]);
        
        register_setting('uscdn_settings', 'uscdn_admin_url', [
            'type' => 'string',
            'default' => '',
            'sanitize_callback' => 'esc_url_raw'
        ]);
        
        // Image CDN Settings
        register_setting('uscdn_settings', 'uscdn_enable_image_cdn', [
            'type' => 'boolean',
            'default' => 0,
            'sanitize_callback' => [$this, 'sanitize_boolean']
        ]);
        
        register_setting('uscdn_settings', 'uscdn_cdn_image_base', [
            'type' => 'string',
            'default' => '',
            'sanitize_callback' => 'esc_url_raw'
        ]);
        
        // Static CDN Settings
        register_setting('uscdn_settings', 'uscdn_enable_static_cdn', [
            'type' => 'boolean',
            'default' => 0,
            'sanitize_callback' => [$this, 'sanitize_boolean']
        ]);
        
        register_setting('uscdn_settings', 'uscdn_cdn_static_base', [
            'type' => 'string',
            'default' => '',
            'sanitize_callback' => 'esc_url_raw'
        ]);
        
        // Link CDN Settings
        register_setting('uscdn_settings', 'uscdn_enable_link_cdn', [
            'type' => 'boolean',
            'default' => 0,
            'sanitize_callback' => [$this, 'sanitize_boolean']
        ]);
        
        register_setting('uscdn_settings', 'uscdn_cdn_link_base', [
            'type' => 'string',
            'default' => '',
            'sanitize_callback' => 'esc_url_raw'
        ]);
        
        // Advanced Settings
        register_setting('uscdn_settings', 'uscdn_analytics_enabled', [
            'type' => 'boolean',
            'default' => 1,
            'sanitize_callback' => [$this, 'sanitize_boolean']
        ]);
        
        register_setting('uscdn_settings', 'uscdn_cache_enabled', [
            'type' => 'boolean',
            'default' => 1,
            'sanitize_callback' => [$this, 'sanitize_boolean']
        ]);
        
        register_setting('uscdn_settings', 'uscdn_custom_source_hosts', [
            'type' => 'string',
            'default' => '',
            'sanitize_callback' => 'sanitize_textarea_field'
        ]);
        
        // Dynamic Content Rewriting
        register_setting('uscdn_settings', 'uscdn_enable_dynamic_rewriting', [
            'type' => 'boolean',
            'default' => 0,
            'sanitize_callback' => [$this, 'sanitize_boolean']
        ]);
        
        register_setting('uscdn_settings', 'uscdn_dynamic_rewriting_auto_rules', [
            'type' => 'boolean',
            'default' => 1,
            'sanitize_callback' => [$this, 'sanitize_boolean']
        ]);
        
        register_setting('uscdn_settings', 'uscdn_dynamic_rewriting_aggressive', [
            'type' => 'boolean',
            'default' => 0,
            'sanitize_callback' => [$this, 'sanitize_boolean']
        ]);
        
        register_setting('uscdn_settings', 'uscdn_dynamic_rewrite_rules', [
            'type' => 'string',
            'default' => '',
            'sanitize_callback' => 'sanitize_textarea_field'
        ]);
    }
    
    public function sanitize_boolean($value) {
        return (int) !!$value;
    }
    
    /**
     * Get a setting value
     */
    public static function get($key, $default = null) {
        $value = get_option('uscdn_' . $key, $default);
        return apply_filters('uscdn_setting_' . $key, $value);
    }
    
    /**
     * Update a setting value
     */
    public static function set($key, $value) {
        return update_option('uscdn_' . $key, $value);
    }
    
    /**
     * Get all CDN bases
     */
    public static function get_cdn_bases() {
        return [
            'img' => rtrim(self::get('cdn_image_base', ''), '/'),
            'static' => rtrim(self::get('cdn_static_base', ''), '/'),
            'link' => rtrim(self::get('cdn_link_base', ''), '/'),
        ];
    }
    
    /**
     * Get source hosts
     */
    public static function get_source_hosts() {
        $hosts = [];
        
        // Current request host
        $h1 = $_SERVER['HTTP_HOST'] ?? '';
        if ($h1) $hosts[] = strtolower($h1);
        
        // Home URL host
        $h2 = parse_url(home_url(), PHP_URL_HOST);
        if ($h2) $hosts[] = strtolower($h2);
        
        // Site URL host
        $h3 = parse_url(site_url(), PHP_URL_HOST);
        if ($h3) $hosts[] = strtolower($h3);
        
        // Custom hosts from settings
        $custom = self::get('custom_source_hosts', '');
        if (!empty($custom)) {
            $custom_hosts = array_map('trim', explode("\n", $custom));
            foreach ($custom_hosts as $host) {
                if (!empty($host)) {
                    $hosts[] = strtolower($host);
                }
            }
        }
        
        $hosts = array_values(array_unique(array_filter($hosts)));
        return apply_filters('uscdn_source_hosts', $hosts);
    }
    
    /**
     * Check if setup is completed
     */
    public static function is_setup_completed() {
        return (bool) self::get('setup_completed', 0);
    }
}
