<?php
/**
 * CDN Validator
 * Tests and validates CDN configurations
 */

if (!defined('ABSPATH')) {
    exit;
}

class USCDN_Validator {
    
    private static $instance = null;
    
    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    /**
     * Test a CDN URL
     */
    public static function test_cdn_url($cdn_url, $test_path = '/wp-includes/js/jquery/jquery.min.js') {
        if (empty($cdn_url)) {
            return [
                'success' => false,
                'message' => 'CDN URL is empty'
            ];
        }
        
        $test_url = rtrim($cdn_url, '/') . $test_path;
        
        $response = wp_remote_get($test_url, [
            'timeout' => 10,
            'sslverify' => true
        ]);
        
        if (is_wp_error($response)) {
            return [
                'success' => false,
                'message' => 'Connection failed: ' . $response->get_error_message()
            ];
        }
        
        $status_code = wp_remote_retrieve_response_code($response);
        
        if ($status_code === 200) {
            return [
                'success' => true,
                'message' => 'CDN is working correctly',
                'status_code' => $status_code,
                'response_time' => $this->get_response_time($response)
            ];
        } else {
            return [
                'success' => false,
                'message' => 'Unexpected status code: ' . $status_code,
                'status_code' => $status_code
            ];
        }
    }
    
    /**
     * Test all configured CDNs
     */
    public static function test_all_cdns() {
        $results = [];
        
        // Test Image CDN
        if (USCDN_Settings::get('enable_image_cdn')) {
            $img_cdn = USCDN_Settings::get('cdn_image_base');
            $results['image'] = self::test_cdn_url($img_cdn, '/wp-content/uploads/test.jpg');
        }
        
        // Test Static CDN
        if (USCDN_Settings::get('enable_static_cdn')) {
            $static_cdn = USCDN_Settings::get('cdn_static_base');
            $results['static'] = self::test_cdn_url($static_cdn);
        }
        
        // Test Link CDN
        if (USCDN_Settings::get('enable_link_cdn')) {
            $link_cdn = USCDN_Settings::get('cdn_link_base');
            $results['link'] = self::test_cdn_url($link_cdn, '/');
        }
        
        return $results;
    }
    
    /**
     * Validate CDN configuration
     */
    public static function validate_config() {
        $errors = [];
        
        // Check Image CDN
        if (USCDN_Settings::get('enable_image_cdn')) {
            $img_cdn = USCDN_Settings::get('cdn_image_base');
            if (empty($img_cdn)) {
                $errors[] = 'Image CDN is enabled but URL is not configured';
            } elseif (!filter_var($img_cdn, FILTER_VALIDATE_URL)) {
                $errors[] = 'Image CDN URL is not valid';
            }
        }
        
        // Check Static CDN
        if (USCDN_Settings::get('enable_static_cdn')) {
            $static_cdn = USCDN_Settings::get('cdn_static_base');
            if (empty($static_cdn)) {
                $errors[] = 'Static CDN is enabled but URL is not configured';
            } elseif (!filter_var($static_cdn, FILTER_VALIDATE_URL)) {
                $errors[] = 'Static CDN URL is not valid';
            }
        }
        
        // Check Link CDN
        if (USCDN_Settings::get('enable_link_cdn')) {
            $link_cdn = USCDN_Settings::get('cdn_link_base');
            if (empty($link_cdn)) {
                $errors[] = 'Link CDN is enabled but URL is not configured';
            } elseif (!filter_var($link_cdn, FILTER_VALIDATE_URL)) {
                $errors[] = 'Link CDN URL is not valid';
            }
        }
        
        return [
            'valid' => empty($errors),
            'errors' => $errors
        ];
    }
    
    /**
     * Get response time from HTTP response
     */
    private static function get_response_time($response) {
        $headers = wp_remote_retrieve_headers($response);
        // This is a simplified version - actual response time would need to be measured
        return 'N/A';
    }
}
